import { Injectable, ConsoleLogger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { InjectRepository } from '@nestjs/typeorm';
import { DuelLogPlayer } from './entities/DuelLogPlayer';
import { FindOptionsWhere, In, Not, Repository } from 'typeorm';
import { MatchResultDto } from './dto/match-result.dto';
import { GenericReturnMessageDto } from 'nicot';

@Injectable()
export class AppService extends ConsoleLogger {
  private dealerUsername =
    this.config.get<string>('DEALER_USERNAME')?.split(',') || [];
  constructor(
    private readonly config: ConfigService,
    @InjectRepository(DuelLogPlayer)
    private readonly duelLogPlayerRepository: Repository<DuelLogPlayer>,
  ) {
    super('app');
  }

  async getMatchResult(where: FindOptionsWhere<DuelLogPlayer>) {
    const [win, total] = await Promise.all([
      this.duelLogPlayerRepository.count({
        where: {
          ...where,
          winner: 1,
        },
      }),
      this.duelLogPlayerRepository.count({
        where,
      }),
    ]);
    const dto = new MatchResultDto();
    dto.win = win || 0;
    dto.total = total || 0;
    dto.lose = dto.total - dto.win;
    return new GenericReturnMessageDto(200, 'success', dto);
  }

  async getMatchResultInv(username: string) {
    const [win, total] = await Promise.all([
      this.duelLogPlayerRepository
        .createQueryBuilder('duelLogPlayer')
        .innerJoin('duelLogPlayer.duelLog', 'duelLog')
        .innerJoin(
          'duelLog.players',
          'dealer',
          'dealer.realName IN (:...dealerUsernames) and dealer.winner = 0 and dealer.lp <= 0 and dealer.score <= 0',
          {
            dealerUsernames: this.dealerUsername,
          },
        )
        .where('duelLogPlayer.realName = :username', { username })
        .andWhere('duelLogPlayer.score >= dealer.score') // -9 and 0 is not win
        .andWhere('duelLogPlayer.lp > 0')
        .select('COUNT(duelLogPlayer.id)', 'win')
        .getRawOne<{ win: string }>(),
      this.duelLogPlayerRepository.count({
        where: {
          realName: username,
        },
      }),
    ]);
    const dto = new MatchResultDto();
    dto.win = parseInt(win.win) || 0;
    dto.total = total || 0;
    dto.lose = dto.total - dto.win;
    return new GenericReturnMessageDto(200, 'success', dto);
  }

  async getDealerMatchResult() {
    return await this.getMatchResult({
      realName: In(this.dealerUsername),
    });
  }

  async getUserMatchResult(username: string) {
    return await this.getMatchResultInv(username);
  }
}
