import { Body, Controller, Get, Ip, Param, ParseArrayPipe, Post, Query, Render, Req, Res, ValidationPipe } from '@nestjs/common';
import { UpdateService } from './update.service';
import { ApiBody, ApiCreatedResponse, ApiOkResponse, ApiOperation, ApiParam, ApiTags } from '@nestjs/swagger';
import { DepotDto } from '../dto/Depot.dto';
import { Request, Response } from 'express';
import { createHash } from 'crypto';

function md5HexOfString(s: string) {
  return createHash('md5').update(s).digest('hex');
}

/**
 * 以“稳定字节串”为准计算 ETag，并发送响应。
 * - 不捕获异常，让 Nest 接管错误处理
 * - 若 If-None-Match 命中，返回 304
 * - 若未显式设置 Cache-Control，则使用传入的默认值
 */
async function stableSend(
  req: Request,
  res: Response,
  body: any | Promise<any>,
  defaultCache = 'public, max-age=31536000, immutable',
  contentType = 'application/json; charset=utf-8'
): Promise<void> {
  const bodyResolved = await body;
  const bodyString = typeof bodyResolved === 'string' ? bodyResolved : JSON.stringify(bodyResolved);
  const hash = md5HexOfString(bodyString);
  const quoted = `"${hash}"`;

  // 协商缓存：If-None-Match 命中则 304
  const inm = req.headers['if-none-match'];
  if (typeof inm === 'string' && inm === quoted) {
    res.setHeader('ETag', quoted);
    if (res.getHeader('Cache-Control') === undefined) {
      res.setHeader('Cache-Control', defaultCache);
    }
    res.status(304).end();
    return;
  }

  // 设置头
  res.setHeader('ETag', quoted);
  if (res.getHeader('Cache-Control') === undefined) {
    res.setHeader('Cache-Control', defaultCache);
  }
  if (!res.getHeader('Content-Type')) {
    res.setHeader('Content-Type', contentType);
  }

  // 发送“刚刚参与了 MD5 计算”的同一份字节串
  res.end(bodyString);
}

@Controller('update')
@ApiTags('update')
export class UpdateController {
  constructor(private readonly updateService: UpdateService) {}

  private async stableReturn<T>(res: Response, prom: Promise<T> | T, cache = 'public, max-age=31536000, immutable') {
    const result = await prom;
    const stringified = JSON.stringify(result);
    const hash = createHash('md5').update(stringified).digest('hex');
    res.setHeader('ETag', `"${hash}"`);
    if (res.getHeader('Cache-Control') === undefined) res.setHeader('Cache-Control', cache);
    res.end(stringified);
    // return result;
  }

  @Get('apps.json')
  @ApiOperation({ summary: '获取 apps.json', description: '懒得解释这是啥了……' })
  async getAppsJson(@Req() req: Request, @Res({ passthrough: true }) res: Response) {
    await stableSend(req, res, this.updateService.getAppsJson(), 'public, max-age=600, stale-while-revalidate=600, stale-if-error=604800');
    return;
  }

  @Get('checksums/:id/:version')
  @ApiOperation({ summary: '获取 app 校验和', description: '是 shasum 的格式' })
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'version', description: 'APP 的版本号' })
  @ApiOkResponse({ type: String })
  // @Header('Cache-Control', 'public, max-age=31536000, immutable') // 可留给 stableSend 兜底
  async getChecksum(
    @Req() req: Request,
    @Res({ passthrough: true }) res: Response,
    @Param('id') id: string,
    @Query(new ValidationPipe({ transform: true })) depot: DepotDto,
    @Param('version') version: string,
  ) {
    await stableSend(req, res, this.updateService.getChecksum(id, depot, version));
    return;
  }

  @Get('metalinks/:id/:version')
  @Render('metalinks')
  @ApiOperation({ summary: '获取 app 完整包 metalink', description: '只包含完整包的' })
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'version', description: 'APP 的版本号' })
  @ApiParam({ name: 'full', description: '强制使用完整包' })
  @ApiOkResponse({ type: String })
  // @Header('Cache-Control', 'public, max-age=31536000, immutable')
  async getFullPackageMetalink(
    @Param('id') id: string,
    @Query(new ValidationPipe({ transform: true })) depot: DepotDto,
    @Param('version') version: string,
    @Ip() ip: string,
    @Query('full') full: string | undefined = undefined
  ) {
    return this.updateService.getFullPackageMetalink(id, depot, version, ip, !!full);
  }

  @Get('single/:path')
  @Render('metalinks')
  @ApiOperation({ summary: '获取单个包 metalink', description: '测试使用' })
  @ApiParam({ name: 'path', description: '路径' })
  @ApiOkResponse({ type: String })
  async getSinglePackageMetalink(@Param('path') path: string, @Ip() ip: string) {
    return this.updateService.getSinglePackageMetalink(path, ip);
  }

  @Post('update/:id/:version')
  @Render('metalinks')
  @ApiOperation({ summary: '获取 app 部分包 metalink', description: '根据文件返回需要下载什么文件' })
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'version', description: 'APP 的版本号' })
  @ApiBody({ type: [String], description: '需要什么文件' })
  @ApiCreatedResponse({ type: String })
  async getPartPackageMetalink(
    @Param('id') id: string,
    @Query(new ValidationPipe({ transform: true })) depot: DepotDto,
    @Param('version') version: string,
    @Body(new ParseArrayPipe()) requestedFiles: string[],
    @Ip() ip: string
  ) {
    //return requestedFiles;
    return this.updateService.getPartPackageMetalink(id, depot, version, requestedFiles, ip);
  }
}
