import {
  Body,
  Controller,
  Delete,
  Get,
  Param,
  Post,
  Query,
  Req,
  UploadedFile,
  UseGuards,
  UseInterceptors,
  ValidationPipe,
} from '@nestjs/common';
import { AppService } from './app.service';
import { ApiBody, ApiConsumes, ApiCreatedResponse, ApiOkResponse, ApiOperation, ApiParam, ApiQuery } from '@nestjs/swagger';
import {
  BlankReturnMessageDto,
  BuildReturnMessageDto,
  GetAppReturnMessageDto,
  ReturnMessageDto,
  StringReturnMessageDto,
} from './dto/ReturnMessage.dto';
import { FetchMyCardUser, MyCardUser } from './utility/mycard-auth';
import { AppsJson } from './utility/apps-json-type';
import { MyCardAppMaintainerGuard } from './my-card-app-maintainer.guard';
import { FileInterceptor } from '@nestjs/platform-express';
import { FileUploadDto } from './dto/FileUpload.dto';
import { AssetsS3Service } from './assets-s3/assets-s3.service';
import Busboy from 'busboy';
import { Request } from 'express';
import { Stream } from 'stream';
import { PackageResult } from './dto/PackageResult.dto';
import { platform } from 'os';
import AppClass = AppsJson.AppClass;
import { DepotDto } from './dto/Depot.dto';

@Controller('api')
export class AppController {
  constructor(private readonly appService: AppService, private readonly s3: AssetsS3Service) {}

  @Get('app')
  @ApiOperation({
    summary: '获取 app',
    description: '管理员可以查询全部的，其他用户可以查属于自己的',
  })
  @ApiQuery({ name: 'id', description: 'app 的 id，没有就是查全部' })
  @ApiOkResponse({ type: GetAppReturnMessageDto })
  getApp(@FetchMyCardUser() user: MyCardUser, @Query('id') id?: string) {
    return this.appService.getApp(user, id);
  }

  @Post('app')
  @ApiOperation({
    summary: '更新 app',
  })
  @ApiBody({ type: AppsJson.AppClass })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  updateApp(@FetchMyCardUser() user: MyCardUser, @Body(new ValidationPipe({ transform: true })) app: AppClass) {
    return this.appService.updateApp(user, app.id, app);
  }

  @Post('assets')
  @ApiOperation({
    summary: '上传附件',
    description: '必须登录用户且必须是管理员或者拥有1个 app 才能上传',
  })
  @UseInterceptors(FileInterceptor('file'))
  @ApiConsumes('multipart/form-data')
  @ApiBody({
    description: 'apps.json 文件',
    type: FileUploadDto,
  })
  @ApiCreatedResponse({ type: StringReturnMessageDto })
  @UseGuards(MyCardAppMaintainerGuard)
  async uploadAssets(@UploadedFile() file: Express.Multer.File) {
    if (!file) {
      throw new BlankReturnMessageDto(400, 'no file').toException();
    }
    const res = await this.s3.uploadAssets(file);
    if (res) {
      return new ReturnMessageDto(201, 'success', res);
    } else {
      throw new BlankReturnMessageDto(500, 'upload fail').toException();
    }
  }

  @Delete('build/:id/:version')
  @ApiOperation({
    summary: '删除打包',
    description: '删除的打包会被彻底删除',
  })
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'version', description: 'APP 的版本号' })
  @ApiQuery({ type: DepotDto, description: 'APP 的类型' })
  async removeBuild(
    @FetchMyCardUser() user: MyCardUser,
    @Param('id') id: string,
    @Query(new ValidationPipe({ transform: true })) depot: DepotDto,
    @Param('version') version: string
  ): Promise<BlankReturnMessageDto> {
    return this.appService.removeBuild(user, id, depot, version);
  }

  @Post('build/:id/:version')
  @ApiOperation({
    summary: '打包文件',
    description: '必须登录用户且必须是管理员或者拥有1个 app 才能上传',
  })
  @ApiConsumes('multipart/form-data')
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'version', description: 'APP 的版本号' })
  @ApiQuery({ type: DepotDto, description: 'APP 的类型' })
  @ApiBody({
    description: 'app 的 tar.zst 文件',
    type: FileUploadDto,
  })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  async makeBuild(
    @FetchMyCardUser() user: MyCardUser,
    @Param('id') id: string,
    @Query(new ValidationPipe({ transform: true })) depot: DepotDto,
    @Param('version') version: string,
    @Req() req: Request
  ) {
    let busboy: busboy.Busboy;
    try {
      busboy = new Busboy({ headers: req.headers });
    } catch (e) {
      throw new BlankReturnMessageDto(400, `Creation failed: ${e.toString()}`).toException();
    }
    const packagePromise = new Promise<ReturnMessageDto<PackageResult>>((resolve, reject) => {
      let gotFile = false;
      busboy.on('file', async (fieldname, fileStream, filename, encoding, mimetype) => {
        fileStream.pause();
        if (fieldname !== 'file') {
          reject(new BlankReturnMessageDto(400, 'invalid field').toException());
          return;
        }
        gotFile = true;
        // console.log(`got file ${fieldname}`);
        try {
          resolve(await this.appService.makeBuild(user, fileStream, id, depot, version));
        } catch (e) {
          reject(e);
        }
      });
      busboy.on('error', () => {
        reject(new BlankReturnMessageDto(500, 'upload error').toException());
      });
      busboy.on('finish', () => {
        //console.log(`finished receiving file`);
        if (!gotFile) {
          reject(new BlankReturnMessageDto(400, 'no file').toException());
        }
        //resolve();
      });
    });
    req.pipe(busboy);
    return packagePromise;
  }
}
