import { ConsoleLogger, Injectable } from '@nestjs/common';
import { InjectConnection } from '@nestjs/typeorm';
import { Connection, IsNull, Not, SelectQueryBuilder } from 'typeorm';
import { App } from '../entities/App.entity';
import { DepotDto } from '../dto/Depot.dto';
import { Build } from '../entities/Build.entity';
import { BlankReturnMessageDto } from '../dto/ReturnMessage.dto';
import { ArchiveType } from '../entities/Archive.entity';
import { PackageS3Service } from '../package-s3/package-s3.service';

@Injectable()
export class UpdateService extends ConsoleLogger {
  private readonly cdnUrl: string;
  constructor(
    @InjectConnection('app')
    private db: Connection,
    packageS3: PackageS3Service
  ) {
    super('update');
    this.cdnUrl = packageS3.cdnUrl;
  }

  async getAppsJson() {
    return (await this.db.getRepository(App).find({ where: { appData: Not(IsNull()), isDeleted: false } })).map((a) => a.appData);
  }

  private async getBuild(id: string, depotDto: DepotDto, version: string, extraQuery?: (query: SelectQueryBuilder<Build>) => void) {
    const depotObj = depotDto.toActual;
    const query = this.db
      .getRepository(Build)
      .createQueryBuilder('build')
      .innerJoin('build.depot', 'depot')
      .innerJoin('depot.app', 'app')
      .leftJoinAndSelect('build.archives', 'archive')
      .where('app.id = :id', { id })
      .andWhere('app.isDeleted = false')
      .andWhere('depot.platform = :platform')
      .andWhere('depot.arch = :arch')
      .andWhere('depot.locale = :locale')
      .andWhere('build.version = :version', { version })
      .setParameters(depotObj);
    if (extraQuery) {
      extraQuery(query);
    }
    const build = await query.getOne();
    if (!build) {
      throw new BlankReturnMessageDto(404, 'Build not found').toException();
    }
    return build;
  }

  async getChecksum(id: string, depotDto: DepotDto, version: string) {
    const build = await this.getBuild(id, depotDto, version);
    return {
      files: Object.entries(build.checksum)
        // .filter(([name, hash]) => file.length && hash !== null)
        .map(([name, hash]) => ({ name, hash })),
    };
  }

  async getFullPackageMetalink(id: string, depotDto: DepotDto, version: string) {
    const build = await this.getBuild(id, depotDto, version, (qb) =>
      qb.andWhere('archive.role = :fullRole', { fullRole: ArchiveType.Full })
    );
    return {
      cdnUrl: this.cdnUrl,
      archives: build.archives,
    };
  }
}
