import fs from 'fs';
import path from 'path';
import child_process from 'child_process';
import os from 'os';
import util from 'util';

import _ from 'lodash';
import { ConfigService } from '@nestjs/config';
import internal, { Stream } from 'stream';
import { PackageResult } from '../dto/PackageResult.dto';
import { PackageS3Service } from '../package-s3/package-s3.service';
import readdirp from 'readdirp';

import { ConsoleLogger, Injectable } from '@nestjs/common';
import { createHash } from 'crypto';
import { Archive, ArchiveType } from '../entities/Archive.entity';
import { Build } from '../entities/Build.entity';

export interface FileWithHash {
  file: readdirp.EntryInfo;
  hash: string;
}

export class ArchiveTask {
  readonly path: string;
  constructor(public readonly role: ArchiveType, public readonly files: FileWithHash[], public readonly altFiles?: string[]) {
    this.path = createHash('sha512')
      .update(files.map((f) => `${f.file.path}${f.hash}`).join(''))
      .digest('hex');
  }

  get archiveFullPath() {
    return `${this.path}.tar.gz`;
  }

  get filePaths() {
    return this.altFiles || this.exactFilePaths;
  }

  get exactFilePaths() {
    return this.files.map((f) => f.file.path);
  }

  get archive() {
    const archive = new Archive();
    archive.path = this.path;
    archive.role = this.role;
    archive.files = this.exactFilePaths;
    return archive;
  }

  addToTask(archiveTasks: ArchiveTask[]) {
    if (archiveTasks.some((t) => t.path === this.path)) {
      return;
    }
    archiveTasks.push(this);
  }
}

@Injectable()
export class PackagerService extends ConsoleLogger {
  bucket_max = 10 * 1024 ** 2;
  bucket_enter = 1 * 1024 ** 2;
  packagerWorkingDirectory: string;

  constructor(private s3: PackageS3Service, config: ConfigService) {
    super('packager');
    this.bucket_max = (parseInt(config.get('PACKAGE_BUCKET_MAX')) || 10) * 1024 ** 2;
    this.bucket_enter = (parseInt(config.get('PACKAGE_BUCKET_ENTER')) || 1) * 1024 ** 2;
    this.packagerWorkingDirectory = config.get('PACKAGE_WORKING_DIRECTORY') || os.tmpdir();
  }

  async build(
    stream: NodeJS.ReadableStream,
    pathPrefix?: string,
    lastBuildChecksums: Record<string, string>[] = []
  ): Promise<PackageResult> {
    this.log(`Start packaging.`);
    const root = await fs.promises.mkdtemp(path.join(this.packagerWorkingDirectory, 'mycard-console-'));
    // const tarballRoot = await fs.promises.mkdtemp(path.join(this.packagerWorkingDirectory, 'mycard-console-tarball-'));
    let extractRoot = root;
    if (pathPrefix) {
      extractRoot = path.join(root, pathPrefix);
      await fs.promises.mkdir(extractRoot, { recursive: true });
    }
    try {
      this.log(`Extracting package to ${extractRoot}.`);
      stream.resume();
      //stream.on('data', (data) => this.log(`data => ${data}`));
      await this.spawnAsync('tar', ['-zxf', '-'], { cwd: extractRoot }, stream);

      this.log(`Package extracted to ${extractRoot}.`);

      //const packagesSequence: string[][] = [];
      const entries = await readdirp.promise(root, { alwaysStat: true, type: 'files_directories' });
      const [directories, files] = _.partition(entries, (item) => item.stats.isDirectory());

      // checksum
      const checksum = await this.checksum(
        root,
        directories.map((d) => d.path),
        files.map((f) => f.path)
      );
      const archiveTasks: ArchiveTask[] = [];

      const filesWithHash: FileWithHash[] = files.map((f) => ({ file: f, hash: checksum[f.path] }));

      // 整包
      new ArchiveTask(ArchiveType.Full, filesWithHash, await fs.promises.readdir(root)).addToTask(archiveTasks);

      // 更新包
      for (const lastChecksum of lastBuildChecksums) {
        const changedFiles = filesWithHash.filter((f) => !lastChecksum[f.file.path] || lastChecksum[f.file.path] !== f.hash);
        if (changedFiles.length) {
          new ArchiveTask(ArchiveType.Update, changedFiles).addToTask(archiveTasks);
        }
      }

      // 散包
      const buckets: Record<string, [FileWithHash[], number]> = {};
      for (const file of filesWithHash) {
        if (file.file.stats.size < this.bucket_enter) {
          const extname = path.extname(file.file.basename);
          buckets[extname] ??= [[], 0];
          const bucket = buckets[extname];
          if (bucket[1] + file.file.stats.size >= this.bucket_max) {
            new ArchiveTask(ArchiveType.Part, bucket[0]).addToTask(archiveTasks);
            bucket[0] = [];
            bucket[1] = 0;
          } else {
            bucket[0].push(file);
            bucket[1] += file.file.stats.size;
          }
        } else {
          new ArchiveTask(ArchiveType.Part, [file]).addToTask(archiveTasks);
        }
      }
      for (const bucket of Object.values(buckets)) {
        if (bucket[0].length) {
          new ArchiveTask(ArchiveType.Part, bucket[0]).addToTask(archiveTasks);
        }
      }

      const packages = await Promise.all(archiveTasks.map((t) => this.archive(root, t))); // 这个 await 过后，checksum 和 打包上传都已经跑完了

      //for (let i = 0; i < packagesSequence.length; ++i) {
      //  packages[gotPackages[i]] = packagesSequence[i];
      //}

      // this.log({ checksum, packages });

      return new PackageResult(checksum, packages);
    } catch (e) {
      throw e;
    } finally {
      await fs.promises.rm(root, { recursive: true });
      // await fs.promises.rm(tarballRoot, { recursive: true });
    }
  }

  async checksum(root: string, directories: string[], files: string[]): Promise<Record<string, string>> {
    const { stdout } = await util.promisify(child_process.execFile)('sha256sum', files, { maxBuffer: 1 * 1024 ** 2, cwd: root });
    return Object.fromEntries([
      ['.', ''],
      ...directories.map((d) => [d, '']),
      ...stdout.split('\n').map((line) => line.split('  ', 2).reverse()),
    ]);
  }

  async archive(root: string, archiveTask: ArchiveTask): Promise<Archive> {
    const archive = archiveTask.archive;
    const archiveName = archiveTask.archiveFullPath;
    const existing = await this.s3.fileExists(archiveName);
    if (existing) {
      archive.size = existing.Size;
      return archive;
    }
    const files = archiveTask.filePaths;
    this.log(`Packaging archive ${archiveName} with ${files.length} files.`);
    const stream = new Stream.Writable();
    const child = child_process.spawn('tar', ['-zcvf', '-'].concat(files), {
      cwd: root,
    });
    const childPromise = new Promise<void>((resolve, reject) => {
      child.on('exit', (code) => {
        if (code == 0) {
          resolve();
        } else {
          reject(code);
        }
      });
      child.on('error', (error) => {
        reject(error);
      });
    });
    const uploadPromise = this.s3.uploadStream(archiveName, child.stdout, {
      ContentType: 'application/tar+gzip',
    });
    const [, { object }] = await Promise.all([childPromise, uploadPromise]);
    archive.size = object.Size;
    return archive;
  }

  private spawnAsync(command: string, args: string[], options: child_process.SpawnOptions, stdinStream?: NodeJS.ReadableStream) {
    return new Promise<void>((resolve, reject) => {
      const child = child_process.spawn(command, args, options);
      if (stdinStream) {
        stdinStream.pipe(child.stdin);
      }
      child.on('exit', (code) => {
        if (code == 0) {
          resolve();
        } else {
          reject(code);
        }
      });
      child.on('error', (error) => {
        reject(error);
      });
    });
  }
}
