import { ConsoleLogger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { ListObjectsCommand, PutObjectCommand, PutObjectCommandInput, S3Client } from '@aws-sdk/client-s3';
import { createHash } from 'crypto';
import internal from 'stream';
import Path from 'path';

export class S3Service extends ConsoleLogger {
  private readonly bucket: string;
  private readonly prefix: string;
  private readonly cdnUrl: string;
  private readonly s3: S3Client;

  private getConfig(field: string) {
    return this.config.get(`${this.servicePrefix}_${field}`) || this.config.get(field);
  }

  constructor(private servicePrefix: string, private config: ConfigService) {
    super(`${servicePrefix} s3`);
    this.bucket = this.getConfig('S3_BUCKET');
    this.prefix = this.getConfig('S3_PREFIX');
    this.cdnUrl = this.getConfig('S3_CDN_URL') || `${this.getConfig('S3_ENDPOINT')}/${this.bucket}${this.prefix ? `/${this.prefix}` : ''}`;
    this.s3 = new S3Client({
      credentials: {
        accessKeyId: this.getConfig('S3_KEY'),
        secretAccessKey: this.getConfig('S3_SECRET'),
      },
      region: this.getConfig('S3_REGION') || 'us-west-1',
      endpoint: this.getConfig('S3_ENDPOINT'),
      forcePathStyle: true,
    });
  }

  async listObjects(path: string) {
    const command = new ListObjectsCommand({
      Bucket: this.bucket,
      Prefix: this.getPathWithPrefix(path),
    });
    return this.s3.send(command);
  }

  async fileExists(path: string) {
    const objects = await this.listObjects(path);
    // this.log(objects);
    return objects.Contents && objects.Contents.some((obj) => obj.Key === this.getPathWithPrefix(path));
  }

  private getPathWithPrefix(filename: string) {
    return this.prefix ? `${this.prefix}/${filename}` : filename;
  }

  async uploadAssets(file: Express.Multer.File) {
    const fileSuffix = Path.extname(file.originalname);
    const hash = createHash('sha512').update(file.buffer).digest('hex');
    const path = fileSuffix ? `${hash}.${fileSuffix}` : hash;
    const downloadUrl = `${this.cdnUrl}/${path}`;
    try {
      if (await this.fileExists(path)) {
        // already uploaded
        return downloadUrl;
      } else {
        return this.uploadFile(path, file.buffer, { ContentType: file.mimetype });
      }
    } catch (e) {
      this.error(`Failed to upload assets ${path}: ${e.toString()}`);
      return null;
    }
  }

  async uploadFile(path: string, content: Buffer | internal.Readable, extras: Partial<PutObjectCommandInput> = {}) {
    await this.s3.send(
      new PutObjectCommand({
        Bucket: this.bucket,
        Key: this.getPathWithPrefix(path),
        Body: content,
        ...extras,
      })
    );
    return `${this.cdnUrl}/${path}`;
  }
}
