import { Body, Controller, Get, Param, Post, Query, Req, UploadedFile, UseGuards, UseInterceptors, ValidationPipe } from '@nestjs/common';
import { AppService } from './app.service';
import { ApiBody, ApiConsumes, ApiCreatedResponse, ApiOkResponse, ApiOperation, ApiParam, ApiQuery } from '@nestjs/swagger';
import { BlankReturnMessageDto, GetAppReturnMessageDto, ReturnMessageDto, StringReturnMessageDto } from './dto/ReturnMessage.dto';
import { FetchMyCardUser, MyCardUser } from './utility/mycard-auth';
import { AppsJson } from './utility/apps-json-type';
import { MyCardAppMaintainerGuard } from './my-card-app-maintainer.guard';
import { FileInterceptor } from '@nestjs/platform-express';
import { FileUploadDto } from './dto/FileUpload.dto';
import { AssetsS3Service } from './assets-s3/assets-s3.service';
import Busboy from 'busboy';
import { Request } from 'express';
import { Stream } from 'stream';
import { PackageResult } from './dto/PackageResult.dto';
import { platform } from 'os';
import AppClass = AppsJson.AppClass;

@Controller('api')
export class AppController {
  constructor(private readonly appService: AppService, private readonly s3: AssetsS3Service) {}

  @Get('apps.json')
  getAppsJson() {
    return this.appService.getAppsJson();
  }

  @Get('app')
  @ApiOperation({
    summary: '获取 app',
    description: '管理员可以查询全部的，其他用户可以查属于自己的',
  })
  @ApiQuery({ name: 'id', description: 'app 的 id，没有就是查全部' })
  @ApiOkResponse({ type: GetAppReturnMessageDto })
  getApp(@FetchMyCardUser() user: MyCardUser, @Query('id') id?: string) {
    return this.appService.getApp(user, id);
  }

  @Post('app')
  @ApiOperation({
    summary: '更新 app',
  })
  @ApiBody({ type: AppsJson.AppClass })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  updateApp(@FetchMyCardUser() user: MyCardUser, @Body(new ValidationPipe({ transform: true })) app: AppClass) {
    return this.appService.updateApp(user, app.id, app);
  }

  @Post('assets')
  @ApiOperation({
    summary: '上传附件',
    description: '必须登录用户且必须是管理员或者拥有1个 app 才能上传',
  })
  @UseInterceptors(FileInterceptor('file'))
  @ApiConsumes('multipart/form-data')
  @ApiBody({
    description: 'apps.json 文件',
    type: FileUploadDto,
  })
  @ApiCreatedResponse({ type: StringReturnMessageDto })
  @UseGuards(MyCardAppMaintainerGuard)
  async uploadAssets(@UploadedFile() file: Express.Multer.File) {
    if (!file) {
      throw new BlankReturnMessageDto(400, 'no file').toException();
    }
    const res = await this.s3.uploadAssets(file);
    if (res) {
      return new ReturnMessageDto(201, 'success', res);
    } else {
      throw new BlankReturnMessageDto(500, 'upload fail').toException();
    }
  }

  @Post('build/:id/:platform/:locale/:version')
  @ApiOperation({
    summary: '打包文件',
    description: '必须登录用户且必须是管理员或者拥有1个 app 才能上传',
  })
  @ApiConsumes('multipart/form-data')
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'platform', description: 'APP 的 版本号', enum: AppsJson.Platform })
  @ApiParam({ name: 'locale', description: 'APP 的 版本号', enum: AppsJson.Locale })
  @ApiParam({ name: 'version', description: 'APP 的 版本号' })
  @ApiBody({
    description: 'app 的 tar.gz 文件',
    type: FileUploadDto,
  })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  async makeBuild(
    @FetchMyCardUser() user: MyCardUser,
    //@UploadedFile() file: Express.Multer.File,
    @Param('id') id: string,
    @Param('platform') platform: AppsJson.Platform,
    @Param('locale') locale: AppsJson.Locale,
    @Param('version') version: string,
    @Req() req: Request
  ) {
    let busboy: busboy.Busboy;
    try {
      busboy = new Busboy({ headers: req.headers });
    } catch (e) {
      throw new BlankReturnMessageDto(400, `Creation failed: ${e.toString()}`).toException();
    }
    const packagePromise = new Promise<ReturnMessageDto<PackageResult>>((resolve, reject) => {
      let gotFile = false;
      busboy.on('file', async (fieldname, fileStream, filename, encoding, mimetype) => {
        if (fieldname !== 'file') {
          reject(new BlankReturnMessageDto(400, 'invalid field').toException());
          return;
        }
        gotFile = true;
        // console.log(`got file ${fieldname}`);
        const stream = new Stream.Readable().wrap(fileStream);
        try {
          resolve(await this.appService.makeBuild(user, stream, id, platform, locale, version));
        } catch (e) {
          stream.destroy();
          reject(e);
        }
      });
      busboy.on('error', () => {
        reject(new BlankReturnMessageDto(500, 'upload error').toException());
      });
      busboy.on('finish', () => {
        //console.log(`finished receiving file`);
        if (!gotFile) {
          reject(new BlankReturnMessageDto(400, 'no file').toException());
        }
        //resolve();
      });
    });
    req.pipe(busboy);
    return packagePromise;
  }
}
