import { ConsoleLogger, Injectable } from '@nestjs/common';
import { InjectConnection } from '@nestjs/typeorm';
import { Brackets, Connection, IsNull, Not, SelectQueryBuilder } from 'typeorm';
import { App } from '../entities/App.entity';
import { DepotDto } from '../dto/Depot.dto';
import { Build } from '../entities/Build.entity';
import { BlankReturnMessageDto } from '../dto/ReturnMessage.dto';
import { Archive, ArchiveType } from '../entities/Archive.entity';
import { PackageS3Service } from '../package-s3/package-s3.service';
import _ from 'lodash';
import { AppsJson } from '../utility/apps-json-type';
import { MirrorService } from '../mirror/mirror.service';
import Platform = AppsJson.Platform;

@Injectable()
export class UpdateService extends ConsoleLogger {
  private readonly cdnUrl: string;
  constructor(
    @InjectConnection('app')
    private db: Connection,
    packageS3: PackageS3Service,
    private mirror: MirrorService
  ) {
    super('update');
    this.cdnUrl = packageS3.cdnUrl;
  }

  async getAppsJson() {
    const data = (await this.db.getRepository(App).find({ where: { appData: Not(IsNull()), isDeleted: false }, select: ['appData'] })).map(
      (a) => a.displayData
    );
    const appVersionCacheMap = new Map<string, string>();
    for (const _platform of Object.values(AppsJson.Platform)) {
      const platform = _platform as AppsJson.Platform;
      if (platform === 'generic') {
        continue;
      }
      /*const depotDto = new DepotDto();
      depotDto.locale = 'generic';
      depotDto.arch = 'generic';
      depotDto.platform = platform;*/
      // this.log(platform);
      for (const app of data) {
        if (app.version && app.version[platform] === '_latest') {
          // this.log(`Try fetching latest version of ${app.id} ${platform}`);
          const tryCache = appVersionCacheMap.get(app.id);
          if (tryCache) {
            app.version[platform] = tryCache;
          }

          const latestBuild = await this.db
            .getRepository(Build)
            .createQueryBuilder('build')
            .select(['build.version', 'build.id', 'depot.platform'])
            .innerJoin('build.depot', 'depot')
            .where('depot.appId = :appId', { appId: app.id })
            .andWhere(
              new Brackets((qb) =>
                qb
                  .where('depot.platform = :genericPlatform', { genericPlatform: Platform.generic })
                  .orWhere('depot.platform = :platform', { platform })
              )
            )
            .orderBy('build.id', 'DESC')
            .take(1)
            .getOne();
          if (latestBuild) {
            const latestVersion = latestBuild.version;
            if (latestBuild.depot.platform === 'generic') {
              appVersionCacheMap.set(app.id, latestVersion);
            }
            app.version[platform] = latestVersion;
          }
        }
      }
    }
    return data;
  }

  private async getBuild(
    id: string,
    depotDto: DepotDto,
    version: string,
    extraQuery?: (query: SelectQueryBuilder<Build>) => void,
    noErrorExit = false
  ) {
    const query = this.db
      .getRepository(Build)
      .createQueryBuilder('build')
      .innerJoin('build.depot', 'depot')
      .innerJoin('depot.app', 'app')
      .where('app.id = :id', { id })
      .andWhere('app.isDeleted = false')
      .andWhere(depotDto.getQueryBrackets())
      .andWhere('build.version = :version', { version });
    if (extraQuery) {
      extraQuery(query);
    }
    const build = await query.getOne();
    if (!noErrorExit && !build) {
      throw new BlankReturnMessageDto(404, 'Build not found').toException();
    }
    return build;
  }

  private async getArchives(
    id: string,
    depotDto: DepotDto,
    version: string,
    extraQuery?: (query: SelectQueryBuilder<Archive>) => void,
    noErrorExit = false
  ) {
    const query = this.db
      .getRepository(Archive)
      .createQueryBuilder('archive')
      .innerJoin('archive.build', 'build')
      .innerJoin('build.depot', 'depot')
      .innerJoin('depot.app', 'app')
      .where('app.id = :id', { id })
      .andWhere('app.isDeleted = false')
      .andWhere(depotDto.getQueryBrackets())
      .andWhere('build.version = :version', { version });
    if (extraQuery) {
      extraQuery(query);
    }
    const archives = await query.getMany();
    if (!noErrorExit && !archives.length) {
      throw new BlankReturnMessageDto(404, 'Build not found').toException();
    }
    return archives;
  }

  async getChecksum(id: string, depotDto: DepotDto, version: string) {
    const build = await this.getBuild(id, depotDto, version, (qb) => qb.select('build.checksum'));
    return {
      files: Object.entries(build.checksum)
        // .filter(([name, hash]) => file.length && hash !== null)
        .map(([name, hash]) => ({ name, hash })),
    };
  }

  async getFullPackageMetalink(id: string, depotDto: DepotDto, version: string) {
    const archives = await this.getArchives(id, depotDto, version, (qb) =>
      qb.select(['archive.hash', 'archive.path', 'archive.size']).andWhere('archive.role = :fullRole', { fullRole: ArchiveType.Full })
    );
    await this.mirror.lookForArchivesMirror(archives);
    return {
      cdnUrl: this.cdnUrl,
      archives: archives,
    };
  }

  /*
  private getPartArchiveSubset(
    requestedFiles: string[],
    allArchives: Archive[],
    currentSize = 0,
    currentBestSolutionValue = Infinity
  ): [Archive[], number] {
    if (!requestedFiles.length) {
      return [[], currentSize];
    }
    let bestSolution: [Archive[], number] = [null, currentBestSolutionValue];
    for (let i = 0; i < allArchives.length; ++i) {
      const archive = allArchives[i];
      const nextStepSize = currentSize + archive.size;
      if (nextStepSize > bestSolution[1]) {
        // 加一个包就太大了的话，不考虑这个方案了
        continue;
      }
      const remainingFiles = _.difference(requestedFiles, archive.files);
      const remainingArchives = allArchives.slice(i + 1);
      const nextStepResult = this.getPartArchiveSubset(remainingFiles, remainingArchives, nextStepSize, bestSolution[1]);
      if (nextStepResult[0] !== null && nextStepResult[1] < bestSolution[1]) {
        nextStepResult[0].push(archive);
        bestSolution = nextStepResult;
        // this.log(`Got better: ${nextStepResult[0].map((a) => `${a.size}:${a.files.join(',')}`).join('|')} ${nextStepResult[1]}`);
      }
    }
    return bestSolution;
  }
   */

  private getCostOfArchives(archives: Archive[]) {
    const requestCost = archives.length * (0.05 / 10000);
    const trafficCost = _.sumBy(archives, (a) => a.size) * (0.24 / 1024 ** 3);
    return trafficCost + requestCost;
  }

  private pickArchives(archives: Archive[], requestedFiles: string[]) {
    if (!archives.length) {
      return [];
    }
    const aset = archives.map((a) => ({ archive: a, files: new Set(a.files) }));

    const result = new Set<Archive>();

    for (const file of requestedFiles) {
      try {
        result.add(aset.find((a) => a.files.has(file)).archive);
      } catch (e) {
        //throw new BlankReturnMessageDto(404, `File ${file} not found in archives.`).toException();
      }
    }
    return Array.from(result);
  }

  async getPartPackageMetalink(id: string, depotDto: DepotDto, version: string, requestedFiles: string[]) {
    if (!requestedFiles || !requestedFiles.length) {
      throw new BlankReturnMessageDto(400, 'empty files').toException();
    }
    const build = await this.getBuild(
      id,
      depotDto,
      version,
      (qb) => qb.select('build.id')
      //.leftJoin('build.archives', 'archive', 'archive.role = :partRole', { partRole: ArchiveType.Part })
    );
    // console.log(build.archives);
    //let clock = moment();
    //this.log('part 1');
    const tryExactArchiveQuery = this.db
      .getRepository(Archive)
      .createQueryBuilder('archive')
      .select(['archive.hash', 'archive.path', 'archive.size'])
      .where('archive.buildId = :buildId', { buildId: build.id })
      .andWhere('archive.role != :partRole', { partRole: ArchiveType.Part })
      .andWhere(':requestedFiles = archive.files', { requestedFiles })
      /*.addSelect(`array(${qb
            .subQuery()
            .select('file.path')
            .from(ArchiveFile, 'file')
            .where('file.archiveId = archive.id')
            .getQuery()})`, 'allFiles')
    tryExactArchiveQuery
      .andWhere(
        `:requestedFiles = array(${tryExactArchiveQuery
          .subQuery()
          .select('file.path')
          .from(ArchiveFile, 'file')
          .where('file.archiveId = archive.id')
          .getQuery()})`,
        { requestedFiles: requestedFiles }
      )*/
      //  .orderBy('archive.size', 'ASC')
      .limit(1);
    const tryExactArchives = await tryExactArchiveQuery.getMany();
    if (tryExactArchives.length) {
      return {
        cdnUrl: this.cdnUrl,
        archives: tryExactArchives,
      };
    }
    //this.log(`Time used: ${moment().diff(clock, 'seconds')} s`);
    /*clock = moment();
    this.log('part ex');
    const archiveIds: number[] = (
      await this.db
        .createQueryBuilder()
        .select('distinct(file.archiveId)', 'archiveId')
        .from(ArchiveFile, 'file')
        .where('file.path = any(:requestedFiles)', { requestedFiles: requestedFiles })
        .innerJoin('file.archive', 'archive')
        .andWhere('archive.buildId = :buildId', { buildId: build.id })
        .getRawMany()
    ).map((obj) => obj.archiveId);*/
    //this.log(`Time used: ${moment().diff(clock, 'seconds')} s`);
    //clock = moment();
    //this.log('part 2');
    const allPossiblePartArchives = await this.db
      .getRepository(Archive)
      .createQueryBuilder('archive')
      .select(['archive.hash', 'archive.path', 'archive.size', 'archive.files'])
      .where('archive.buildId = :buildId', { buildId: build.id })
      .andWhere('archive.role = :partRole', { partRole: ArchiveType.Part })
      .getMany();
    //.innerJoin('archive.containingFiles', 'file')
    //.andWhere('file.path = any(:requestedFiles)', { requestedFiles: requestedFiles });

    /*
    allPartArchivesQuery.andWhere(
      `archive.id in (${allPartArchivesQuery
        .subQuery()
        .select('distinct(file.archiveId)')
        .from(ArchiveFile, 'file')
        .where('file.archiveId = any(:possibleArchiveIds)', { possibleArchiveIds: build.archives.map((a) => a.id) })
        .andWhere('file.path = any(:requestedFiles)', { requestedFiles: requestedFiles })
        .getQuery()})`
    );
     */

    //const allPartArchives = await allPartArchivesQuery.getMany();
    //this.log(`Time used: ${moment().diff(clock, 'seconds')} s`);
    //clock = moment();
    //this.log('part 3');
    const allPartArchives = this.pickArchives(allPossiblePartArchives, requestedFiles);
    //this.log(`Time used: ${moment().diff(clock, 'seconds')} s`);
    //clock = moment();
    //this.log('part 4');
    const fullArchive = await this.db
      .getRepository(Archive)
      .createQueryBuilder('archive')
      .select(['archive.hash', 'archive.path', 'archive.size'])
      .where('archive.buildId = :buildId', { buildId: build.id })
      .andWhere('archive.role = :fullRole', { fullRole: ArchiveType.Full })
      .limit(1)
      .getOne();

    let archives = allPartArchives;
    if (!allPartArchives.length || (fullArchive && this.getCostOfArchives([fullArchive]) <= this.getCostOfArchives(allPartArchives))) {
      archives = [fullArchive];
    }
    await this.mirror.lookForArchivesMirror(archives);
    //this.log(`Time used: ${moment().diff(clock, 'seconds')} s`);
    return {
      cdnUrl: this.cdnUrl,
      archives,
    };
  }
}
