import { Connection, IsNull, Not, SelectQueryBuilder } from 'typeorm';
import { InjectConnection } from '@nestjs/typeorm';
import { ConsoleLogger, forwardRef, Inject, Injectable } from '@nestjs/common';
import { AppsJson } from './utility/apps-json-type';
import { App } from './entities/App.entity';
import { BlankReturnMessageDto, ReturnMessageDto } from './dto/ReturnMessage.dto';
import { MyCardUser } from './utility/mycard-auth';
import { PackagerService } from './packager/packager.service';
import internal from 'stream';
import { Depot } from './entities/Depot.entity';
import { DepotDto } from './dto/Depot.dto';
import { Build } from './entities/Build.entity';
import { ConfigService } from '@nestjs/config';
import { Archive, ArchiveType } from './entities/Archive.entity';
import { PackageS3Service } from './package-s3/package-s3.service';
import { createHash } from 'crypto';
import { HttpService } from '@nestjs/axios';
import { lastValueFrom } from 'rxjs';
import { MirrorService } from './mirror/mirror.service';
import { AssetsS3Service } from './assets-s3/assets-s3.service';

@Injectable()
export class AppService extends ConsoleLogger {
  private readonly packageVersionTraceCount: number;
  private readonly packageVersionPreserveCount: number;
  constructor(
    @Inject(forwardRef(() => AppService))
    appService: AppService,
    @InjectConnection('app')
    private db: Connection,
    private packager: PackagerService,
    private packageS3: PackageS3Service,
    private assetsS3: AssetsS3Service,
    config: ConfigService,
    private http: HttpService,
    private mirror: MirrorService
  ) {
    super('app');
    // 打包追溯几个版本的更新包
    this.packageVersionTraceCount = parseInt(config.get('PACKAGE_VERSION_TRACE_COUNT')) || 5;
    // 清理掉几个版本之前的部分包
    this.packageVersionPreserveCount = parseInt(config.get('PACKAGE_VERSION_PRESERVE_COUNT')) || 5;
  }

  private async updateResult<T>(f: () => Promise<T>, returnCode = 200) {
    try {
      const result = await f();
      return new ReturnMessageDto<T>(returnCode, 'success', result);
    } catch (e) {
      throw new BlankReturnMessageDto(404, 'Database Fail').toException();
    }
  }

  async migrateFromAppsJson(apps: AppsJson.App[]) {
    const targetApps: App[] = [];
    for (const appData of apps) {
      if (!appData.id) {
        throw new BlankReturnMessageDto(400, `App ${appData.name} is invalid.`).toException();
      }
      const checkExistingApp = await this.db.getRepository(App).findOne({ where: { id: appData.id }, relations: ['history'] });
      //this.error('read');
      if (checkExistingApp) {
        checkExistingApp.updateApp(appData);
        checkExistingApp.isDeleted = false;
        targetApps.push(checkExistingApp);
      } else {
        const app = new App();
        app.id = appData.id;
        app.appData = appData;
        targetApps.push(app);
      }
    }
    //this.error('write');
    return this.updateResult(async () => {
      await this.db.getRepository(App).save(targetApps);
      return;
    }, 201);
  }

  async getApp(user: MyCardUser, id?: string) {
    if (!user) {
      throw new BlankReturnMessageDto(401, 'Needs login').toException();
    }
    const query = this.db.getRepository(App).createQueryBuilder('app').where('app.isDeleted = false');
    if (!user.admin) {
      query.andWhere(':uid = ANY(app.author)', { uid: user.id });
    }
    if (id) {
      query.andWhere('app.id = :id', { id });
    }
    query
      .leftJoinAndSelect('app.history', 'history')
      .leftJoinAndSelect('app.depots', 'depot')
      .leftJoin('depot.builds', 'build')
      .addSelect(['build.id', 'build.version']);
    return new ReturnMessageDto(200, 'success', await query.getMany());
  }

  async isUserCanMaintainApp(user: MyCardUser, id?: string) {
    if (user.admin) {
      return true;
    }
    const query = this.db
      .getRepository(App)
      .createQueryBuilder('app')
      .where('app.isDeleted = false')
      .andWhere(':uid = ANY(app.author)', { uid: user.id });
    if (id) {
      query.andWhere('app.id = :id', { id });
    }
    return (await query.getCount()) > 0;
  }

  async createApp(id: string) {
    let app = await this.db.getRepository(App).findOne({ where: { id }, select: ['id', 'isDeleted'] });
    if (!app) {
      app = new App();
      app.id = id;
    } else {
      if (!app.isDeleted) {
        throw new BlankReturnMessageDto(404, 'App already exists').toException();
      }
      app.isDeleted = false;
    }
    return this.updateResult(() => this.db.getRepository(App).save(app));
  }

  async assignApp(id: string, author: number[]) {
    return this.updateResult(() => this.db.getRepository(App).update({ id }, { author }), 201);
  }

  async setAppPrefix(id: string, prefix: string) {
    return this.updateResult(() => this.db.getRepository(App).update({ id }, { packagePrefix: prefix }), 201);
  }

  async updateApp(user: MyCardUser, id: string, appData: AppsJson.App) {
    if (!user) {
      throw new BlankReturnMessageDto(401, 'Needs login').toException();
    }
    appData.id = id;
    const app = await this.db.getRepository(App).findOne({
      where: { id: appData.id, isDeleted: false },
      relations: ['history'],
      select: ['id', 'author', 'appData'],
    });
    if (!app) {
      throw new BlankReturnMessageDto(404, 'App not found').toException();
    }
    if (!app.isUserCanEditApp(user)) {
      throw new BlankReturnMessageDto(403, 'Permission denied').toException();
    }
    app.updateApp(appData, user.id);
    return this.updateResult(async () => {
      await this.db.getRepository(App).save(app);
      return;
    }, 201);
  }

  private async getOrCreateDepot(app: App, depotDto: DepotDto) {
    const depotOption = depotDto.toActual;
    let depot = await this.db.getRepository(Depot).findOne({ where: { app, ...depotOption } });
    if (!depot) {
      depot = new Depot();
      depot.app = app;
      depot.platform = depotOption.platform;
      depot.locale = depotOption.locale;
      depot.arch = depotOption.arch;
      depot.builds = [];
      depot = await this.db.getRepository(Depot).save(depot);
    }
    return depot;
  }

  private async checkExistingBuild(depot: Depot, version: string) {
    return this.db.getRepository(Build).findOne({ where: { depot, version }, select: ['id'] });
  }

  async makeBuild(user: MyCardUser, stream: NodeJS.ReadableStream, id: string, depotDto: DepotDto, version: string) {
    if (!user) {
      throw new BlankReturnMessageDto(401, 'Needs login').toException();
    }
    //this.log('Build: Checking app.');
    const app = await this.db.getRepository(App).findOne({
      where: { id, isDeleted: false },
      select: ['id', 'packagePrefix', 'author'],
    });
    if (!app) {
      throw new BlankReturnMessageDto(404, 'App not found').toException();
    }
    if (!app.isUserCanEditApp(user)) {
      throw new BlankReturnMessageDto(403, 'Permission denied').toException();
    }
    const depot = await this.getOrCreateDepot(app, depotDto);
    if (await this.checkExistingBuild(depot, version)) {
      throw new BlankReturnMessageDto(404, 'Build exists').toException();
    }
    let build = new Build();
    build.depot = depot;
    build.version = version;
    this.log(`Start packaging ${app.id} ${version} for ${JSON.stringify(depot)}.`);
    try {
      const previousTracingBuildChecksums = (
        await this.db
          .getRepository(Build)
          .find({ where: { depot }, order: { id: 'DESC' }, select: ['id', 'checksum'], take: this.packageVersionTraceCount })
      ).map((b) => b.checksum);
      const result = await this.packager.build(stream, app.packagePrefix, previousTracingBuildChecksums);
      this.log(`Saving package info of ${app.id} ${version} for ${JSON.stringify(depot)}`);
      build.checksum = result.checksum;
      //build.archives = result.archives;
      //build = await this.db.getRepository(Build).save(build);

      await this.db.transaction(async (edb) => {
        this.log(`Saving build info.`);
        build = await edb.getRepository(Build).save(build);
        let archivePot: Archive[] = [];
        let currentSize = 0;
        for (const archive of result.archives) {
          archive.build = build;
          const size = archive.getParamSize();
          if (currentSize > 0 && currentSize + size > 60000) {
            this.log(`Saving ${archivePot.length} archive infos.`);
            await edb.getRepository(Archive).save(archivePot);
            archivePot = [];
            currentSize = 0;
          }
          archivePot.push(archive);
          currentSize += size;
        }
        if (archivePot.length) {
          this.log(`Saving ${archivePot.length} archive infos.`);
          await edb.getRepository(Archive).save(archivePot);
        }
      });

      this.mirror.triggerMirror();
      this.log(`Finished packaging ${app.id} ${version} for ${JSON.stringify(depot)}`);
      return new BlankReturnMessageDto(201, 'success');
    } catch (e) {
      this.error(`Build ${app.id} ${JSON.stringify(depotDto.toActual)} ${build.version} failed: ${e.toString()}`);
      throw new BlankReturnMessageDto(500, 'Build failed').toException();
    }
  }

  private packageReferenceSubQuery(query: SelectQueryBuilder<any>) {
    const subQuery = query
      .subQuery()
      .select('referencingArchive.id')
      .from(Archive, 'referencingArchive')
      .where('referencingArchive.path = archive.path')
      .andWhere('referencingArchive.buildId != archive.buildId');
    query.andWhere(`not exists ${subQuery.getQuery()}`);
  }

  async getPurgeOldArchivePaths() {
    const query = this.db
      .getRepository(Build)
      .createQueryBuilder('unusedBuild')
      .select('distinct(archive.path)', 'pathToPurge')
      .innerJoin('unusedBuild.archives', 'archive')
      .where('archive.role != :latestRole', { latestRole: ArchiveType.Full });

    const subQuery = query
      .subQuery()
      .select('latestBuild.id')
      .from(Build, 'latestBuild')
      .where('latestBuild.depotId = unusedBuild.depotId')
      .orderBy('latestBuild.id', 'DESC')
      .take(this.packageVersionPreserveCount);

    const packageReferenceQuery = query
      .subQuery()
      .select('referencingArchive.id')
      .from(Archive, 'referencingArchive')
      .where('referencingArchive.path = archive.path')
      .andWhere(`referencingArchive.buildId in ${subQuery.getQuery()}`);

    query.andWhere(`unusedBuild.id not in ${subQuery.getQuery()}`).andWhere(`not exists ${packageReferenceQuery.getQuery()}`);
    // this.log(`SQL: ${query.getQueryAndParameters()}`);
    return (await query.getRawMany()).map((s) => `${s.pathToPurge}.tar.zst` as string);
  }

  private async getArchivePathsToPurge(buildId: number) {
    const query = this.db
      .getRepository(Archive)
      .createQueryBuilder('archive')
      .select('distinct(archive.path)', 'pathToPurge')
      .where('archive.buildId = :buildId', { buildId });

    this.packageReferenceSubQuery(query);
    // this.log(`SQL: ${query.getQueryAndParameters()}`);
    return (await query.getRawMany()).map((s) => `${s.pathToPurge}.tar.zst` as string);
  }

  async purgeOldArchives() {
    const paths = await this.getPurgeOldArchivePaths();
    this.log(`Will purge file ${paths.join(',')} (${paths.length}) for old archives.`);
    if (!paths.length) {
      return;
    }
    return this.packageS3.removeObjects(paths);
  }

  private async purgeRelatedArchives(build: Build) {
    const paths = await this.getArchivePathsToPurge(build.id);
    this.log(`Will purge file ${paths.join(',')} (${paths.length}) for build ${build.id} removal.`);
    if (!paths.length) {
      return;
    }
    return this.packageS3.removeObjects(paths);
  }

  async removeBuild(user: MyCardUser, id: string, depotDto: DepotDto, version: string) {
    if (!user) {
      throw new BlankReturnMessageDto(401, 'Needs login').toException();
    }
    const app = await this.db.getRepository(App).findOne({
      where: { id, isDeleted: false },
      select: ['id', 'author'],
    });
    if (!app) {
      throw new BlankReturnMessageDto(404, 'App not found').toException();
    }
    if (!app.isUserCanEditApp(user)) {
      throw new BlankReturnMessageDto(403, 'Permission denied').toException();
    }
    const depot = await this.getOrCreateDepot(app, depotDto);
    const build = await this.checkExistingBuild(depot, version);
    if (!build) {
      throw new BlankReturnMessageDto(404, 'Build not found').toException();
    }
    await this.purgeRelatedArchives(build);
    await this.db.transaction(async (edb) => {
      //const archives = await edb.getRepository(Archive).find({ where: { build } });
      //await edb.getRepository(ArchiveFile).delete({ archive: In(archives) });
      await edb.getRepository(Build).delete(build);
    });
    return new BlankReturnMessageDto(200, 'success');
  }

  async deleteApp(id: string) {
    return this.updateResult(() => this.db.getRepository(App).update({ id }, { isDeleted: true }));
  }

  async lookForExistingArchiveHash(path: string) {
    const [archive] = await this.db
      .getRepository(Archive)
      .find({ where: { path, hash: Not(IsNull()) }, select: ['hash'], order: { id: 'DESC' }, take: 1 });
    if (archive) {
      return archive.hash;
    }
    return null;
    // return this.checkHashFromUrl(`${this.packageS3.cdnUrl}/${path}.tar.zst`);
  }

  async checkHashFromUrl(url: string) {
    this.log(`Downloading ${url} for checking hash.`);
    const { data } = await lastValueFrom(this.http.get<internal.Readable>(url, { responseType: 'stream' }));
    return new Promise<string>((resolve, reject) => {
      const hashObject = createHash('sha256');
      data.on('data', (data) => hashObject.update(data));
      data.on('error', reject);
      data.on('end', () => resolve(hashObject.digest('hex')));
    });
  }

  async getHashForMigrate(fullPath: string): Promise<string> {
    const url = this.packageS3.getCdnUrl(fullPath);
    this.log(`Migrating hash: ${url} `);
    const hash = await this.checkHashFromUrl(url);
    this.log(`Migrated hash: ${url} => ${hash}`);
    return hash;
  }
  async migrateHashes() {
    const archivesToDo = await this.db.getRepository(Archive).find({ where: { hash: IsNull() }, select: ['id', 'hash', 'path'] });
    const tmpMap = new Map<string, string>();
    for (const archive of archivesToDo) {
      if (tmpMap.has(archive.path)) {
        archive.hash = tmpMap.get(archive.path);
      } else {
        archive.hash = await this.getHashForMigrate(archive.archiveFullPath);
        tmpMap.set(archive.path, archive.hash);
      }
    }
    await this.db.getRepository(Archive).save(archivesToDo);
    return new BlankReturnMessageDto(200, 'success');
  }

  async uploadAssets(file: Express.Multer.File) {
    const s3Url = await this.assetsS3.uploadAssets(file);
    if (!s3Url) {
      return null;
    }
    const resultFromMirror = await this.mirror.uploadWithRandomMiddleware({
      url: s3Url,
      size: file.size,
      customMime: file.mimetype,
      // customSuffix: path.extname(file.originalname) || undefined;
    });
    return resultFromMirror ? resultFromMirror.url : s3Url;
  }

  /*
  async migrateFilesField() {
    const archives = await this.db.getRepository(Archive).find({ select: ['id', 'files'], where: { files: IsNull() } });
    await this.db.transaction(async (edb) => {
      for (const a of archives) {
        this.log(`Processing archive ${a.id}.`);
        const fileEnts = await this.db.getRepository(ArchiveFile).find({ select: ['path'], where: { archive: a } });
        this.log(`${fileEnts.length} files found for archive ${a.id}`);
        a.files = fileEnts.map((f) => f.path);
        await edb.getRepository(Archive).save(a);
      }
    });
    this.log(`Done.`);
    return new BlankReturnMessageDto(200, 'success');
  }*/
}
