import fs from 'fs';
import path from 'path';
import child_process from 'child_process';
import os from 'os';
import util from 'util';

import _ from 'lodash';
import { ConfigService } from '@nestjs/config';
import internal from 'stream';
import { PackageResult } from '../dto/PackageResult.dto';
import { PackageS3Service } from '../package-s3/package-s3.service';
import readdirp from 'readdirp';
import { v4 as uuidv4 } from 'uuid';

import { ConsoleLogger, Injectable } from '@nestjs/common';

@Injectable()
export class PackagerService extends ConsoleLogger {
  bucket_max = 10 * 1024 ** 2;
  bucket_enter = 1 * 1024 ** 2;

  constructor(private s3: PackageS3Service, config: ConfigService) {
    super('packager');
    this.bucket_max = (parseInt(config.get('PACKAGE_BUCKET_MAX')) || 10) * 1024 ** 2;
    this.bucket_enter = (parseInt(config.get('PACKAGE_BUCKET_ENTER')) || 1) * 1024 ** 2;
  }

  async build(stream: internal.Readable, pathPrefix?: string): Promise<PackageResult> {
    this.log(`Start packaging.`);
    const root = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'mycard-console-'));
    const tarballRoot = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'mycard-console-tarball-'));
    let extractRoot = root;
    if (pathPrefix) {
      extractRoot = path.join(root, pathPrefix);
      await fs.promises.mkdir(extractRoot, { recursive: true });
    }
    await this.spawnAsync('tar', ['-zxvf', '-'], { cwd: extractRoot, stdio: [stream, 'ignore', 'ignore'] });

    this.log(`Package extracted to ${extractRoot}.`);

    const packages: Record<string, string[]> = {};
    const entries = await readdirp.promise(root, { alwaysStat: true, type: 'files_directories' });
    const [directories, files] = _.partition(entries, (item) => item.stats.isDirectory());

    // checksum
    const checksum = await this.checksum(
      root,
      directories.map((d) => d.path),
      files.map((f) => f.path)
    );
    const promises: Promise<[string, string]>[] = [];

    // 整包
    const archive = `${uuidv4()}.tar.gz`;
    packages[archive] = ['_full'];
    promises.push(this.archive(archive, root, tarballRoot, await fs.promises.readdir(root)));

    // 散包
    const buckets: Record<string, [string[], number]> = {};
    for (const file of files) {
      if (file.stats.size < this.bucket_enter) {
        const extname = path.extname(file.basename);
        buckets[extname] ??= [[], 0];
        const bucket = buckets[extname];
        if (bucket[1] + file.stats.size >= this.bucket_max) {
          const archive = `${uuidv4()}.tar.gz`;
          packages[archive] = bucket[0];
          promises.push(this.archive(archive, root, tarballRoot, bucket[0], checksum));
          bucket[0] = [];
          bucket[1] = 0;
        } else {
          bucket[0].push(file.path);
          bucket[1] += file.stats.size;
        }
      } else {
        const archive = `${uuidv4()}.tar.gz`;
        packages[archive] = [file.path];
        promises.push(this.archive(archive, root, tarballRoot, [file.path], checksum));
      }
    }
    for (const bucket of Object.values(buckets)) {
      if (bucket[0].length) {
        const archive = `${uuidv4()}.tar.gz`;
        packages[archive] = bucket[0];
        promises.push(this.archive(archive, root, tarballRoot, bucket[0], checksum));
      }
    }

    // TODO: 更新包

    const gotPackages = await Promise.all(promises); // 这个 await 过后，checksum 和 打包上传都已经跑完了

    for (const differentPackages of gotPackages.filter((p) => p[0] !== p[1])) {
      const [originalPackage, gotPackage] = differentPackages;
      packages[gotPackage] = packages[originalPackage];
      delete packages[originalPackage];
    }

    this.log({ checksum, packages });
    await fs.promises.rm(root, { recursive: true });
    await fs.promises.rm(tarballRoot, { recursive: true });
    return new PackageResult(checksum, packages);
  }

  async checksum(root: string, directories: string[], files: string[]): Promise<Record<string, string>> {
    const { stdout } = await util.promisify(child_process.execFile)('sha256sum', files, { maxBuffer: 1 * 1024 ** 2, cwd: root });
    return Object.fromEntries([
      ['.', ''],
      ...directories.map((d) => [d, '']),
      ...stdout.split('\n').map((line) => line.split('  ', 2).reverse()),
    ]);
  }

  async archive(
    archive: string = `${uuidv4()}.tar.gz`,
    root: string,
    tarballRoot: string,
    files: string[],
    checksum: Record<string, string> = {}
  ): Promise<[string, string]> {
    const archivePath = path.join(tarballRoot, archive);
    await this.spawnAsync('tar', ['-zcvf', archivePath].concat(files), {
      cwd: root,
    });
    const fileSize = (await fs.promises.stat(archivePath)).size;
    await this.s3.uploadFile(archive, fs.createReadStream(archivePath), { ContentType: 'application/tar+gzip', ContentLength: fileSize });
    return [archive, archive];
  }

  private spawnAsync(command: string, args: string[], options: child_process.SpawnOptions) {
    return new Promise<void>((resolve, reject) => {
      const child = child_process.spawn(command, args, options);
      child.on('exit', (code) => {
        if (code == 0) {
          resolve();
        } else {
          reject(code);
        }
      });
      child.on('error', (error) => {
        reject(error);
      });
    });
  }
}
