import {
  BadRequestException,
  Body,
  Controller,
  Get,
  Param,
  Post,
  Query,
  UploadedFile,
  UseGuards,
  UseInterceptors,
  ValidationPipe,
} from '@nestjs/common';
import { AppService } from './app.service';
import { ApiBody, ApiConsumes, ApiCreatedResponse, ApiOkResponse, ApiOperation, ApiParam, ApiQuery } from '@nestjs/swagger';
import { BlankReturnMessageDto, GetAppReturnMessageDto, ReturnMessageDto, StringReturnMessageDto } from './dto/ReturnMessage.dto';
import { FetchMyCardUser, MyCardUser } from './utility/mycard-auth';
import { AppsJson } from './utility/apps-json-type';
import { MyCardAppMaintainerGuard } from './my-card-app-maintainer.guard';
import { FileInterceptor } from '@nestjs/platform-express';
import { FileUploadDto } from './dto/FileUpload.dto';
import AppClass = AppsJson.AppClass;
import { AssetsS3Service } from './assets-s3/assets-s3.service';
import { MulterDirectEngine } from './packager/MulterStreamEngine';
import * as fs from 'fs';

@Controller('api')
export class AppController {
  constructor(private readonly appService: AppService, private readonly s3: AssetsS3Service) {}

  @Get('apps.json')
  getAppsJson() {
    return this.appService.getAppsJson();
  }

  @Get('app')
  @ApiOperation({
    summary: '获取 app',
    description: '管理员可以查询全部的，其他用户可以查属于自己的',
  })
  @ApiQuery({ name: 'id', description: 'app 的 id，没有就是查全部' })
  @ApiOkResponse({ type: GetAppReturnMessageDto })
  getApp(@FetchMyCardUser() user: MyCardUser, @Query('id') id?: string) {
    return this.appService.getApp(user, id);
  }

  @Post('app')
  @ApiOperation({
    summary: '更新 app',
  })
  @ApiBody({ type: AppsJson.AppClass })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  updateApp(@FetchMyCardUser() user: MyCardUser, @Body(new ValidationPipe({ transform: true })) app: AppClass) {
    return this.appService.updateApp(user, app.id, app);
  }

  @Post('assets')
  @ApiOperation({
    summary: '上传附件',
    description: '必须登录用户且必须是管理员或者拥有1个 app 才能上传',
  })
  @UseInterceptors(FileInterceptor('file'))
  @ApiConsumes('multipart/form-data')
  @ApiBody({
    description: 'apps.json 文件',
    type: FileUploadDto,
  })
  @ApiCreatedResponse({ type: StringReturnMessageDto })
  @UseGuards(MyCardAppMaintainerGuard)
  async uploadAssets(@UploadedFile() file: Express.Multer.File) {
    if (!file) {
      throw new BlankReturnMessageDto(400, 'no file').toException();
    }
    const res = await this.s3.uploadAssets(file);
    if (res) {
      return new ReturnMessageDto(201, 'success', res);
    } else {
      throw new BlankReturnMessageDto(500, 'upload fail').toException();
    }
  }

  @Post('build/:id/:platform/:locale/:version')
  @ApiOperation({
    summary: '打包文件',
    description: '必须登录用户且必须是管理员或者拥有1个 app 才能上传',
  })
  @UseInterceptors(FileInterceptor('file', { storage: new MulterDirectEngine() }))
  @ApiConsumes('multipart/form-data')
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'platform', description: 'APP 的 版本号', enum: AppsJson.Platform })
  @ApiParam({ name: 'locale', description: 'APP 的 版本号', enum: AppsJson.Locale })
  @ApiParam({ name: 'version', description: 'APP 的 版本号' })
  /*@ApiBody({
    description: 'app 的 tar.gz 文件',
    type: FileUploadDto,
  })*/
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  async makeBuild(
    @FetchMyCardUser() user: MyCardUser,
    //@UploadedFile() file: Express.Multer.File,
    @Param('id') id: string,
    @Param('platform') platform: AppsJson.Platform,
    @Param('locale') locale: AppsJson.Locale,
    @Param('version') version: string
  ) {
    /*console.log(file.stream);
    if (!file) {
      throw new BlankReturnMessageDto(400, 'no file').toException();
    }*/
    return this.appService.makeBuild(user, fs.createReadStream('/tmp/test1.tar.gz'), id, platform, locale, version);
  }
}
