import fs from 'fs';
import path from 'path';
import child_process from 'child_process';
import os from 'os';
import util from 'util';

import _ from 'lodash';
import { ConfigService } from '@nestjs/config';
import { PackageResult } from '../dto/PackageResult.dto';
import { PackageS3Service } from '../package-s3/package-s3.service';
import readdirp from 'readdirp';

import { ConsoleLogger, forwardRef, Inject, Injectable } from '@nestjs/common';
import { Archive, ArchiveType } from '../entities/Archive.entity';
import { AppService } from '../app.service';
import { createHash } from 'crypto';
import PQueue from 'p-queue';
import { LockService } from 'src/lock/lock.service';
import { InjectRedis, Redis } from '@nestjs-modules/ioredis';
import BetterLock from 'better-lock';

interface FileWithHash {
  file: readdirp.EntryInfo;
  hash: string;
}

const ARG_SIZE_LIMIT = 30000;

class ArchiveTask {
  readonly path: string;
  constructor(public readonly role: ArchiveType, public readonly files: FileWithHash[], public readonly altFiles?: string[]) {
    this.path = createHash('sha512')
      .update(files.map((f) => `${f.file.path}${f.hash}`).join(''))
      .digest('hex');
  }

  get archiveFullPath() {
    return `${this.path}.tar.zst`;
  }

  get filePaths() {
    return this.altFiles || this.exactFilePaths;
  }

  get exactFilePaths() {
    return this.files.map((f) => f.file.path);
  }

  get archive() {
    const archive = new Archive();
    archive.path = this.path;
    archive.role = this.role;
    archive.files = this.exactFilePaths;
    //archive.containingFiles = this.exactFilePaths.map((filePath) => ArchiveFile.fromPath(filePath));
    return archive;
  }

  addToTask(archiveTasks: ArchiveTask[], skipDupCheck?: boolean) {
    if (!skipDupCheck && this.role !== ArchiveType.Part && archiveTasks.some((t) => t.path === this.path)) {
      return;
    }
    archiveTasks.push(this);
  }
}

class Bucket {
  files: FileWithHash[] = [];
  size = 0;

  empty() {
    this.files = [];
    this.size = 0;
  }

  addFile(file: FileWithHash) {
    this.files.push(file);
    this.size += file.file.stats.size;
  }

  canFit(file: FileWithHash, limit: number): boolean {
    return this.size + file.file.stats.size <= limit;
  }
}

@Injectable()
export class PackagerService extends ConsoleLogger {
  bucket_max = 10 * 1024 ** 2;
  bucket_enter = 1024 ** 2;
  noGatherExts = new Set<string>();
  packagerWorkingDirectory: string;

  // private uploadLock = new Set<string>();
  // private hashCache = new Cache<string, string>();

  constructor(
    @Inject(forwardRef(() => AppService)) private readonly appService: AppService,
    private s3: PackageS3Service,
    config: ConfigService,
    // private redlock: LockService,
    @InjectRedis() private readonly redis: Redis
  ) {
    super('packager');
    this.bucket_max = (parseInt(config.get('PACKAGE_BUCKET_MAX')) || 10) * 1024 ** 2;
    this.bucket_enter = (parseInt(config.get('PACKAGE_BUCKET_ENTER')) || 1) * 1024 ** 2;
    this.packagerWorkingDirectory = config.get('PACKAGE_WORKING_DIRECTORY') || os.tmpdir();
    const noGatherExts = config.get('PACKAGE_NO_GATHER_EXTS');
    if (noGatherExts) {
      this.noGatherExts = new Set(noGatherExts.split(','));
    }
  }

  async build(
    stream: NodeJS.ReadableStream,
    pathPrefix?: string,
    lastBuildChecksums: Record<string, string>[] = []
  ): Promise<PackageResult> {
    this.log(`Start packaging.`);
    const root = await fs.promises.mkdtemp(path.join(this.packagerWorkingDirectory, 'mycard-console-'));
    // const tarballRoot = await fs.promises.mkdtemp(path.join(this.packagerWorkingDirectory, 'mycard-console-tarball-'));
    let extractRoot = root;
    if (pathPrefix) {
      extractRoot = path.join(root, pathPrefix);
      await fs.promises.mkdir(extractRoot, { recursive: true });
    }
    try {
      this.log(`Extracting package to ${extractRoot}.`);
      stream.resume();
      //stream.on('data', (data) => this.log(`data => ${data}`));
      await this.spawnAsync('tar', ['--zstd', '-xf', '-'], { cwd: extractRoot }, stream);

      this.log(`Package extracted to ${extractRoot}.`);

      //const packagesSequence: string[][] = [];
      const entries = await readdirp.promise(root, { alwaysStat: true, type: 'files_directories' });
      const [directories, files] = _.partition(entries, (item) => item.stats.isDirectory());

      this.log(`Calculating checksums.`);

      // checksum
      const checksum = await this.checksum(
        root,
        directories.map((d) => d.path),
        files.map((f) => f.path)
      );
      const archiveTasks: ArchiveTask[] = [];

      const filesWithHash: FileWithHash[] = files.map((file) => ({ file, hash: checksum[file.path] }));

      // 整包
      new ArchiveTask(ArchiveType.Full, filesWithHash, await fs.promises.readdir(root)).addToTask(archiveTasks);

      if (files.length > 1) {
        // 整包（512M 一包）
        const cloudflareLimit = 450 * 1024 ** 2; // Cloudflare 的限制是 512M，留点余地
        const totalFileSize = filesWithHash.reduce((sum, f) => sum + f.file.stats.size, 0);
        if (totalFileSize > cloudflareLimit) {
          this.log(`Total file size is ${totalFileSize}, which is larger than ${cloudflareLimit}. Will create fulls archives.`);
          const sortedFiles = _.sortBy(filesWithHash, (f) => -f.file.stats.size); // 按照文件大小降序排序，优先打包大文件
          // create multiple fulls archives, with each archive containing files up to cloudflareLimit
          const buckets: Bucket[] = [];
          for (const file of sortedFiles) {
            const useBucket = buckets.find((b) => b.canFit(file, cloudflareLimit));
            if (useBucket) {
              useBucket.addFile(file);
            } else {
              const newBucket = new Bucket();
              newBucket.addFile(file);
              buckets.push(newBucket);
            }
          }
          if (buckets.length > 1) {
            for (const bucket of buckets) {
              new ArchiveTask(ArchiveType.Fulls, bucket.files).addToTask(archiveTasks);
            }
            this.log(`Created ${buckets.length} fulls archives.`);
          } else {
            this.log(`Only 1 fulls archive created, skipping.`);
          }
        }

        // 更新包
        for (const lastChecksum of lastBuildChecksums) {
          const changedFiles = filesWithHash.filter((f) => !lastChecksum[f.file.path] || lastChecksum[f.file.path] !== f.hash);
          if (changedFiles.length) {
            new ArchiveTask(ArchiveType.Update, changedFiles).addToTask(archiveTasks);
          }
        }
      }

      const pendingPartTasks: ArchiveTask[] = [];

      // 散包
      const buckets: Record<string, Bucket> = {};
      for (const file of filesWithHash) {
        const extname = path.extname(file.file.basename);
        if (file.file.stats.size < this.bucket_enter && !this.noGatherExts.has(extname)) {
          buckets[extname] ??= new Bucket();
          const bucket = buckets[extname];
          if (!bucket.canFit(file, this.bucket_max)) {
            new ArchiveTask(ArchiveType.Part, bucket.files).addToTask(pendingPartTasks, true);
            bucket.empty();
          }
          bucket.addFile(file);
        } else {
          new ArchiveTask(ArchiveType.Part, [file]).addToTask(pendingPartTasks, true);
        }
      }
      for (const bucket of Object.values(buckets)) {
        if (bucket.files.length) {
          new ArchiveTask(ArchiveType.Part, bucket.files).addToTask(pendingPartTasks, true);
        }
      }

      if (pendingPartTasks.length > 1) {
        for (const task of pendingPartTasks) {
          task.addToTask(archiveTasks);
        }
      } else {
        this.log(`Skipping part archives because only 1 part archive.`);
      }

      const packages = await Promise.all(archiveTasks.map((t) => this.archive(root, t))); // 这个 await 过后，checksum 和 打包上传都已经跑完了

      //for (let i = 0; i < packagesSequence.length; ++i) {
      //  packages[gotPackages[i]] = packagesSequence[i];
      //}

      // this.log({ checksum, packages });
      this.log(`Package finished.`);
      return new PackageResult(checksum, packages);
    } catch (e) {
      throw e;
    } finally {
      await fs.promises.rm(root, { recursive: true });
      // await fs.promises.rm(tarballRoot, { recursive: true });
    }
  }

  async checksum(root: string, directories: string[], files: string[]): Promise<Record<string, string>> {
    // 通常系统参数限制在 128KB 左右，保守一点设为 32KB

    function chunkFiles(files: string[], limit = ARG_SIZE_LIMIT): string[][] {
      const chunks: string[][] = [];
      let currentChunk: string[] = [];
      let currentSize = 0;

      for (const file of files) {
        const size = Buffer.byteLength(file) + 1; // +1 for space
        if (currentSize + size > limit && currentChunk.length > 0) {
          chunks.push(currentChunk);
          currentChunk = [];
          currentSize = 0;
        }
        currentChunk.push(file);
        currentSize += size;
      }

      if (currentChunk.length > 0) {
        chunks.push(currentChunk);
      }

      return chunks;
    }

    const fileChunks = chunkFiles(files);

    const chunkPromises = fileChunks.map((chunk) =>
      util
        .promisify(child_process.execFile)('sha256sum', chunk, {
          cwd: root,
          maxBuffer: 1 * 1024 ** 3,
        })
        .then(({ stdout }) =>
          stdout
            .trim()
            .split('\n')
            .map((line) => line.split('  ', 2).reverse())
        )
    );

    const results = (await Promise.all(chunkPromises)).flat();

    return Object.fromEntries([['.', ''], ...directories.map((d) => [d, '']), ...results]);
  }

  private lock = new BetterLock();

  async archive(root: string, archiveTask: ArchiveTask): Promise<Archive> {
    const archive = archiveTask.archive;
    const archiveName = archiveTask.archiveFullPath;
    const existing = await this.s3.fileExists(archiveName);
    if (existing) {
      const hash = await this.appService.lookForExistingArchiveHash(archiveTask.path);
      if (hash) {
        archive.hash = hash;
        archive.size = existing.ContentLength;
        this.log(`Archive ${archiveName} exists, skipping.`);
        return archive;
      }
    }
    return this.lock.acquire([`archive:${archiveTask.path}`], async () => this.archiveProcess(root, archiveTask));
  }

  private archiveQueue = new PQueue({ concurrency: parseInt(process.env.PACKAGE_COCURRENCY) || os.cpus().length });

  private async archiveProcess(root: string, archiveTask: ArchiveTask, retry = 0): Promise<Archive> {
    const archive = archiveTask.archive;
    const archiveName = archiveTask.archiveFullPath;
    const existing = await this.s3.fileExists(archiveName);
    if (existing) {
      const hash = await this.appService.lookForExistingArchiveHash(archiveTask.path);
      if (hash) {
        archive.hash = hash;
        archive.size = existing.ContentLength;
        this.log(`Archive ${archiveName} exists, skipping.`);
        return archive;
      }
    }

    this.log(`Will archive ${archiveName}.`);

    return this.archiveQueue.add(async () => {
      const files = archiveTask.filePaths;
      this.log(`Packaging archive ${archiveName} with ${archiveTask.exactFilePaths.length} files.`);
      let tmpFilename: string;

      let fileParams = files;

      const shouldUseFileList = (files: string[], limit = ARG_SIZE_LIMIT): boolean => {
        const totalLength = files.reduce((sum, f) => sum + Buffer.byteLength(f) + 1, 0); // +1 for space/newline
        return totalLength > limit;
      };

      if (shouldUseFileList(files)) {
        this.warn(`Too many files in archive ${archiveName}, using file list.`);
        tmpFilename = path.join(this.packagerWorkingDirectory, `filelist-${archiveName}.txt`);
        await fs.promises.writeFile(tmpFilename, files.join('\n'), 'utf8');
        fileParams = ['--files-from', tmpFilename];
      }

      const child = child_process.spawn('tar', ['--zstd', '-cf', '-', ...fileParams], {
        cwd: root,
      });
      const childPromise = new Promise<void>((resolve, reject) => {
        child.on('exit', (code) => {
          if (code == 0) {
            resolve();
          } else {
            reject(code);
          }
        });
        child.on('error', (error) => {
          reject(error);
        });
      });
      const hashObject = createHash('sha256');
      // let length = 0;
      child.stdout.on('data', (chunk) => {
        // length += chunk.length;
        // this.log(`Received ${length} bytes of archive ${archiveName}.`);
        hashObject.update(chunk);
      });

      const uploadStream = child.stdout;

      try {
        /* if (files.length > 1000) {
          this.warn(`Too many files in archive ${archiveName}, using tmp file.`);
          // minio would skew the stream if it's too slow

          // use a tmp file to put the stream
          tmpFilename = path.join(this.packagerWorkingDirectory, `tmp-${archiveName}.tar.zst`);
          // put the stream to the tmp file
          const tmpStream = fs.createWriteStream(tmpFilename);
          uploadStream.pipe(tmpStream);
          // wait for the stream to finish
          await new Promise((resolve, reject) => {
            tmpStream.on('finish', resolve);
            tmpStream.on('error', reject);
          });
          // open the tmp file as a new stream
          uploadStream = fs.createReadStream(tmpFilename);
        }*/

        const uploadPromise = this.s3.uploadStream(archiveName, uploadStream, {
          ContentType: 'application/tar+zstd',
        });
        const [, { object }] = await Promise.all([childPromise, uploadPromise]);
        archive.hash = hashObject.digest('hex');
        await this.redis.set(`hash:${archive.path}`, archive.hash, 'EX', 60 * 60 * 24);
        archive.size = object.Size;
        this.log(`Finished archiving ${archiveName}.`);
        return archive;
      } catch (e) {
        this.error(`Failed to archive ${archiveName}: ${e.toString()}`);
        if (retry < 3) {
          this.warn(`Retrying archive ${archiveName}.`);
          return this.archiveProcess(root, archiveTask, retry + 1);
        }
        throw e;
      } finally {
        if (tmpFilename) {
          await fs.promises.rm(tmpFilename);
        }
      }
    });
  }

  private spawnAsync(command: string, args: string[], options: child_process.SpawnOptions, stdinStream?: NodeJS.ReadableStream) {
    return new Promise<void>((resolve, reject) => {
      const child = child_process.spawn(command, args, options);
      if (stdinStream) {
        stdinStream.pipe(child.stdin);
      }
      child.on('exit', (code) => {
        if (code == 0) {
          resolve();
        } else {
          reject(code);
        }
      });
      child.on('error', (error) => {
        reject(error);
      });
    });
  }
}
