import { Column, Entity, Index, ManyToOne, PrimaryGeneratedColumn } from 'typeorm';
import { Build } from './Build.entity';
import { TimeBase } from './TimeBase.entity';
import { ArchiveMirror } from './ArchiveMirror.entity';
import { BigintTransformer } from 'src/utility/bigint';

export enum ArchiveType {
  Full = 'full',
  Update = 'update',
  Part = 'part',
  Fulls = 'fulls',
}

@Entity()
export class Archive extends TimeBase {
  @PrimaryGeneratedColumn({ type: 'int8' })
  id: number;

  @Column('varchar', { length: 256, array: true, default: [] })
  files: string[];

  //@OneToMany(() => ArchiveFile, (file) => file.archive, { cascade: true })
  //containingFiles: ArchiveFile[];

  @Index()
  @Column('varchar', { length: 128 })
  path: string;

  @Column('varchar', { length: 64, nullable: true })
  hash: string;

  @Column('bigint', { unsigned: true, transformer: new BigintTransformer() })
  size: number;

  @ManyToOne((type) => Build, (build) => build.archives, { onDelete: 'CASCADE' })
  build: Build;

  @Index()
  @Column({ type: 'enum', enum: ArchiveType })
  role: ArchiveType;

  // should not be a relation
  mirrors: string[];

  get archiveFullPath() {
    return `${this.path}.tar.zst`;
  }

  getParamSize() {
    return 5 + (this.files?.length || 0);
  }

  url: string;
  backupUrl: string;

  fillUrl(cdnUrl: string, cdnUrlOversize: string, addBackup = false) {
    const overSize = this.size > 536870912;
    if (overSize) {
      this.url = `${cdnUrlOversize}/${this.archiveFullPath}`;
    } else {
      this.url = `${cdnUrl}/${this.archiveFullPath}`;
      if (addBackup) {
        this.backupUrl = `${cdnUrlOversize}/${this.archiveFullPath}`;
      }
    }
  }
}
