import { ConsoleLogger, Injectable } from '@nestjs/common';
import { HttpService } from '@nestjs/axios';
import Address from 'ipaddr.js';
import { lastValueFrom } from 'rxjs';
import { Interval } from '@nestjs/schedule';

@Injectable()
export class ChnrouteService extends ConsoleLogger {
  constructor(private http: HttpService) {
    super('ChnrouteService');
  }

  private misakaUrl = 'https://raw.githubusercontent.com/misakaio/chnroutes2/master/chnroutes.txt';
  private chinaIpRnages: [Address.IPv4, number][] = [];

  async onApplicationBootstrap() {
    this.updateChnroute().then();
  }

  @Interval(3600000)
  async updateChnroute() {
    try {
      const { data } = await lastValueFrom(
        this.http.get<string>(this.misakaUrl, {
          responseType: 'text',
        })
      );
      const ranges = data
        .trim()
        .split('\n')
        .map((line) => line.trim())
        .filter((line) => line && !line.startsWith('#'));
      this.chinaIpRnages = ranges.map((range) => Address.parseCIDR(range) as [Address.IPv4, number]);
      this.log(`update chnroute success: ${this.chinaIpRnages.length}`);
    } catch (e) {
      this.error(`update chnroute error: ${e.message}`);
    }
  }

  isChina(ip: string) {
    if (ip === '::1') return false;
    if (this.chinaIpRnages.length) {
      const ipObj = Address.parse(ip);
      if (ipObj.kind() === 'ipv6') {
        return false; // IPv6 is not supported in this service
      }
      return this.chinaIpRnages.some(([range, mask]) => {
        return ipObj.match(range, mask);
      });
    } else {
      return false;
    }
  }
}
