import fs from 'fs';
import path from 'path';
import child_process from 'child_process';
import os from 'os';
import util from 'util';

import _ from 'lodash';
import { ConfigService } from '@nestjs/config';
import internal from 'stream';
import { PackageResult } from '../dto/PackageResult.dto';
import { PackageS3Service } from '../package-s3/package-s3.service';
import readdirp from 'readdirp';

import { ConsoleLogger, Injectable } from '@nestjs/common';
import { createHash } from 'crypto';

export interface FileWithHash {
  file: readdirp.EntryInfo;
  hash: string;
}

@Injectable()
export class PackagerService extends ConsoleLogger {
  bucket_max = 10 * 1024 ** 2;
  bucket_enter = 1 * 1024 ** 2;

  constructor(private s3: PackageS3Service, config: ConfigService) {
    super('packager');
    this.bucket_max = (parseInt(config.get('PACKAGE_BUCKET_MAX')) || 10) * 1024 ** 2;
    this.bucket_enter = (parseInt(config.get('PACKAGE_BUCKET_ENTER')) || 1) * 1024 ** 2;
  }

  async build(stream: internal.Readable, pathPrefix?: string): Promise<PackageResult> {
    this.log(`Start packaging.`);
    const root = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'mycard-console-'));
    const tarballRoot = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'mycard-console-tarball-'));
    let extractRoot = root;
    if (pathPrefix) {
      extractRoot = path.join(root, pathPrefix);
      await fs.promises.mkdir(extractRoot, { recursive: true });
    }
    this.log(`Extracting package to ${extractRoot}.`);
    await this.spawnAsync('tar', ['-zxf', '-'], { cwd: extractRoot }, stream);

    this.log(`Package extracted to ${extractRoot}.`);

    const packagesSequence: string[][] = [];
    const entries = await readdirp.promise(root, { alwaysStat: true, type: 'files_directories' });
    const [directories, files] = _.partition(entries, (item) => item.stats.isDirectory());

    // checksum
    const checksum = await this.checksum(
      root,
      directories.map((d) => d.path),
      files.map((f) => f.path)
    );
    const promises: Promise<string>[] = [];

    const filesWithHash: FileWithHash[] = files.map((f) => ({ file: f, hash: checksum[f.path] }));

    // 整包
    packagesSequence.push(files.map((f) => f.path));
    promises.push(this.archive(root, tarballRoot, filesWithHash, await fs.promises.readdir(root)));

    // 散包
    const buckets: Record<string, [FileWithHash[], number]> = {};
    for (const file of filesWithHash) {
      if (file.file.stats.size < this.bucket_enter) {
        const extname = path.extname(file.file.basename);
        buckets[extname] ??= [[], 0];
        const bucket = buckets[extname];
        if (bucket[1] + file.file.stats.size >= this.bucket_max) {
          packagesSequence.push(bucket[0].map((f) => f.file.path));
          promises.push(this.archive(root, tarballRoot, bucket[0]));
          bucket[0] = [];
          bucket[1] = 0;
        } else {
          bucket[0].push(file);
          bucket[1] += file.file.stats.size;
        }
      } else {
        packagesSequence.push([file.file.path]);
        promises.push(this.archive(root, tarballRoot, [file]));
      }
    }
    for (const bucket of Object.values(buckets)) {
      if (bucket[0].length) {
        packagesSequence.push(bucket[0].map((f) => f.file.path));
        promises.push(this.archive(root, tarballRoot, bucket[0]));
      }
    }

    // TODO: 更新包

    const gotPackages = await Promise.all(promises); // 这个 await 过后，checksum 和 打包上传都已经跑完了

    const packages: Record<string, string[]> = {};
    for (let i = 0; i < packagesSequence.length; ++i) {
      packages[gotPackages[i]] = packagesSequence[i];
    }

    // this.log({ checksum, packages });
    await fs.promises.rm(root, { recursive: true });
    await fs.promises.rm(tarballRoot, { recursive: true });
    return new PackageResult(checksum, packages, gotPackages[0]);
  }

  async checksum(root: string, directories: string[], files: string[]): Promise<Record<string, string>> {
    const { stdout } = await util.promisify(child_process.execFile)('sha256sum', files, { maxBuffer: 1 * 1024 ** 2, cwd: root });
    return Object.fromEntries([
      ['.', ''],
      ...directories.map((d) => [d, '']),
      ...stdout.split('\n').map((line) => line.split('  ', 2).reverse()),
    ]);
  }

  async archive(root: string, tarballRoot: string, files: FileWithHash[], altFiles?: string[]): Promise<string> {
    const archive =
      createHash('sha512')
        .update(files.map((f) => `${f.file.path}${f.hash}`).join(''))
        .digest('hex') + '.tar.gz';
    if (await this.s3.fileExists(archive)) {
      return archive;
    }
    const archivePath = path.join(tarballRoot, archive);
    this.log(`Packaging archive ${archivePath} with ${files.length} files.`);
    await this.spawnAsync('tar', ['-zcvf', archivePath].concat(altFiles || files.map((f) => f.file.path)), {
      cwd: root,
    });
    const fileSize = (await fs.promises.stat(archivePath)).size;
    await this.s3.uploadFile(archive, fs.createReadStream(archivePath), { ContentType: 'application/tar+gzip', ContentLength: fileSize });
    return archive;
  }

  private spawnAsync(
    command: string,
    args: string[],
    options: child_process.SpawnOptions,
    stdinStream?: internal.Readable,
    stdoutStream?: internal.Writable,
    stderrStream?: internal.Writable
  ) {
    return new Promise<void>((resolve, reject) => {
      const child = child_process.spawn(command, args, options);
      if (stdinStream) {
        stdinStream.pipe(child.stdin);
      }
      if (stdoutStream) {
        child.stdout.pipe(stdoutStream);
      }
      if (stderrStream) {
        child.stderr.pipe(stderrStream);
      }
      child.on('exit', (code) => {
        if (code == 0) {
          resolve();
        } else {
          reject(code);
        }
      });
      child.on('error', (error) => {
        reject(error);
      });
    });
  }
}
