import { Body, Controller, Get, Header, Ip, Param, ParseArrayPipe, Post, Query, Render, Res, ValidationPipe } from '@nestjs/common';
import { UpdateService } from './update.service';
import { ApiBody, ApiCreatedResponse, ApiOkResponse, ApiOperation, ApiParam, ApiTags } from '@nestjs/swagger';
import { DepotDto } from '../dto/Depot.dto';
import { Response } from 'express';
import { createHash } from 'crypto';

@Controller('update')
@ApiTags('update')
export class UpdateController {
  constructor(private readonly updateService: UpdateService) {}
  @Get('apps.json')
  @ApiOperation({ summary: '获取 apps.json', description: '懒得解释这是啥了……' })
  async getAppsJson(@Res({ passthrough: true }) res: Response) {
    const result = await this.updateService.getAppsJson();
    this.addETag(res, JSON.stringify(result));
    res.setHeader('Cache-Control', 'public, max-age=600, stale-while-revalidate=600, stale-if-error=604800');
    return result;
  }

  private addETag(res: Response, data: any) {
    const hash = createHash('md5').update(JSON.stringify(data)).digest('hex');
    res.setHeader('ETag', `"${hash}"`);
  }

  private async cacheResult<T>(res: Response, prom: Promise<T>) {
    const result = await prom;
    this.addETag(res, result);
    if (res.getHeader('Cache-Control') === undefined) res.setHeader('Cache-Control', 'public, max-age=31536000, immutable');
    return result;
  }

  @Get('checksums/:id/:version')
  @Render('checksums')
  @ApiOperation({ summary: '获取 app 校验和', description: '是 shasum 的格式' })
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'version', description: 'APP 的版本号' })
  @ApiOkResponse({ type: String })
  // @Header('Cache-Control', 'public, max-age=31536000, immutable')
  async getChecksum(
    @Param('id') id: string,
    @Query(new ValidationPipe({ transform: true })) depot: DepotDto,
    @Param('version') version: string,
    @Res({ passthrough: true }) res: Response
  ) {
    return this.cacheResult(res, this.updateService.getChecksum(id, depot, version));
  }

  @Get('metalinks/:id/:version')
  @Render('metalinks')
  @ApiOperation({ summary: '获取 app 完整包 metalink', description: '只包含完整包的' })
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'version', description: 'APP 的版本号' })
  @ApiParam({ name: 'full', description: '强制使用完整包' })
  @ApiOkResponse({ type: String })
  // @Header('Cache-Control', 'public, max-age=31536000, immutable')
  async getFullPackageMetalink(
    @Param('id') id: string,
    @Query(new ValidationPipe({ transform: true })) depot: DepotDto,
    @Param('version') version: string,
    @Ip() ip: string,
    @Query('full') full: string | undefined = undefined
  ) {
    return this.updateService.getFullPackageMetalink(id, depot, version, ip, !!full);
  }

  @Get('single/:path')
  @Render('metalinks')
  @ApiOperation({ summary: '获取单个包 metalink', description: '测试使用' })
  @ApiParam({ name: 'path', description: '路径' })
  @ApiOkResponse({ type: String })
  async getSinglePackageMetalink(@Param('path') path: string, @Ip() ip: string) {
    return this.updateService.getSinglePackageMetalink(path, ip);
  }

  @Post('update/:id/:version')
  @Render('metalinks')
  @ApiOperation({ summary: '获取 app 部分包 metalink', description: '根据文件返回需要下载什么文件' })
  @ApiParam({ name: 'id', description: 'APP 的 id' })
  @ApiParam({ name: 'version', description: 'APP 的版本号' })
  @ApiBody({ type: [String], description: '需要什么文件' })
  @ApiCreatedResponse({ type: String })
  async getPartPackageMetalink(
    @Param('id') id: string,
    @Query(new ValidationPipe({ transform: true })) depot: DepotDto,
    @Param('version') version: string,
    @Body(new ParseArrayPipe()) requestedFiles: string[],
    @Ip() ip: string
  ) {
    //return requestedFiles;
    return this.updateService.getPartPackageMetalink(id, depot, version, requestedFiles, ip);
  }
}
