import { ConsoleLogger, Injectable } from '@nestjs/common';
import { InjectConnection } from '@nestjs/typeorm';
import { Connection, IsNull, Not, SelectQueryBuilder } from 'typeorm';
import { App } from '../entities/App.entity';
import { DepotDto } from '../dto/Depot.dto';
import { Build } from '../entities/Build.entity';
import { BlankReturnMessageDto } from '../dto/ReturnMessage.dto';
import { Archive, ArchiveType } from '../entities/Archive.entity';
import { PackageS3Service } from '../package-s3/package-s3.service';
import _ from 'lodash';

@Injectable()
export class UpdateService extends ConsoleLogger {
  private readonly cdnUrl: string;
  constructor(
    @InjectConnection('app')
    private db: Connection,
    packageS3: PackageS3Service
  ) {
    super('update');
    this.cdnUrl = packageS3.cdnUrl;
  }

  async getAppsJson() {
    return (await this.db.getRepository(App).find({ where: { appData: Not(IsNull()), isDeleted: false } })).map((a) => a.appData);
  }

  private async getBuild(
    id: string,
    depotDto: DepotDto,
    version: string,
    extraQuery?: (query: SelectQueryBuilder<Build>) => void,
    noErrorExit = false
  ) {
    const query = this.db
      .getRepository(Build)
      .createQueryBuilder('build')
      .innerJoin('build.depot', 'depot')
      .innerJoin('depot.app', 'app')
      .where('app.id = :id', { id })
      .andWhere('app.isDeleted = false')
      .andWhere(depotDto.getQueryBrackets())
      .andWhere('build.version = :version', { version });
    if (extraQuery) {
      extraQuery(query);
    }
    const build = await query.getOne();
    if (!noErrorExit && !build) {
      throw new BlankReturnMessageDto(404, 'Build not found').toException();
    }
    return build;
  }

  private async getArchives(
    id: string,
    depotDto: DepotDto,
    version: string,
    extraQuery?: (query: SelectQueryBuilder<Archive>) => void,
    noErrorExit = false
  ) {
    const query = this.db
      .getRepository(Archive)
      .createQueryBuilder('archive')
      .innerJoin('archive.build', 'build')
      .innerJoin('build.depot', 'depot')
      .innerJoin('depot.app', 'app')
      .where('app.id = :id', { id })
      .andWhere('app.isDeleted = false')
      .andWhere(depotDto.getQueryBrackets())
      .andWhere('build.version = :version', { version });
    if (extraQuery) {
      extraQuery(query);
    }
    const archives = await query.getMany();
    if (!noErrorExit && !archives.length) {
      throw new BlankReturnMessageDto(404, 'Build not found').toException();
    }
    return archives;
  }

  async getChecksum(id: string, depotDto: DepotDto, version: string) {
    const build = await this.getBuild(id, depotDto, version);
    return {
      files: Object.entries(build.checksum)
        // .filter(([name, hash]) => file.length && hash !== null)
        .map(([name, hash]) => ({ name, hash })),
    };
  }

  async getFullPackageMetalink(id: string, depotDto: DepotDto, version: string) {
    const archives = await this.getArchives(id, depotDto, version, (qb) =>
      qb.andWhere('archive.role = :fullRole', { fullRole: ArchiveType.Full })
    );
    return {
      cdnUrl: this.cdnUrl,
      archives: archives,
    };
  }

  private getPartArchiveSubset(
    requestedFiles: string[],
    allArchives: Archive[],
    currentSize = 0,
    currentBestSolutionValue = Infinity
  ): [Archive[], number] {
    /*
    this.log(
      `requested: ${requestedFiles.join(',')} remaining: ${allArchives
        .map((a) => `${a.size}:${a.files.join(',')}`)
        .join('|')} currentSize: ${currentSize} currentBestSolutionValue: ${currentBestSolutionValue}`
    );
     */
    if (!requestedFiles.length) {
      return [[], currentSize];
    }
    let bestSolution: [Archive[], number] = [null, currentBestSolutionValue];
    for (let i = 0; i < allArchives.length; ++i) {
      const archive = allArchives[i];
      const nextStepSize = currentSize + archive.size;
      if (nextStepSize > bestSolution[1]) {
        // 加一个包就太大了的话，不考虑这个方案了
        continue;
      }
      const remainingFiles = _.difference(requestedFiles, archive.files);
      const remainingArchives = allArchives.slice(i + 1);
      const nextStepResult = this.getPartArchiveSubset(remainingFiles, remainingArchives, nextStepSize, bestSolution[1]);
      if (nextStepResult[0] !== null && nextStepResult[1] < bestSolution[1]) {
        nextStepResult[0].push(archive);
        bestSolution = nextStepResult;
        // this.log(`Got better: ${nextStepResult[0].map((a) => `${a.size}:${a.files.join(',')}`).join('|')} ${nextStepResult[1]}`);
      }
    }
    return bestSolution;
  }

  async getPartPackageMetalink(id: string, depotDto: DepotDto, version: string, requestedFiles: string[]) {
    const tryExactArchives = await this.getArchives(
      id,
      depotDto,
      version,
      (qb) =>
        qb
          .andWhere('archive.role != :partRole', { partRole: ArchiveType.Part })
          .andWhere(':requestedFiles = archive.files', { requestedFiles: requestedFiles })
          .orderBy('archive.size', 'ASC')
          .limit(1),
      true
    );
    if (tryExactArchives.length) {
      return {
        cdnUrl: this.cdnUrl,
        archives: tryExactArchives,
      };
    }
    const allArchives = await this.getArchives(id, depotDto, version, (qb) =>
      qb.andWhere(':requestedFiles && archive.files', { requestedFiles: requestedFiles }).orderBy('archive.size', 'ASC')
    );
    /*for (const archive of allArchives) {
      archive.files = _.intersection(archive.files, requestedFiles);
    }*/
    const [archives] = this.getPartArchiveSubset(_.uniq(requestedFiles), allArchives);
    if (!archives) {
      throw new BlankReturnMessageDto(404, 'Some files of this build not found').toException();
    }
    return {
      cdnUrl: this.cdnUrl,
      archives,
    };
  }
}
