import { Body, Controller, Delete, Param, Post, Put, UploadedFile, UseGuards, UseInterceptors, ValidationPipe } from '@nestjs/common';
import { ApiBody, ApiConsumes, ApiCreatedResponse, ApiOkResponse, ApiOperation, ApiTags } from '@nestjs/swagger';
import { MyCardAdminGuard } from '../my-card-admin.guard';
import { FileInterceptor } from '@nestjs/platform-express';
import { FileUploadDto } from '../dto/FileUpload.dto';
import { AppsJson } from '../utility/apps-json-type';
import { AppService } from '../app.service';
import { BlankReturnMessageDto, ReturnMessageDto } from '../dto/ReturnMessage.dto';
import { AssignAppDto } from '../dto/AssignApp.dto';
import { AppPrefixDto } from '../dto/AppPrefix.dto';

@Controller('api/admin')
@ApiTags('admin')
@UseGuards(MyCardAdminGuard)
export class AdminController {
  constructor(private readonly appService: AppService) {}

  @Post('migrate/apps.json')
  @ApiOperation({ summary: '迁移 apps', description: '上传一个 apps.json 文件' })
  @UseInterceptors(FileInterceptor('file'))
  @ApiConsumes('multipart/form-data')
  @ApiBody({
    description: 'apps.json 文件',
    type: FileUploadDto,
  })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  async migrateApps(@UploadedFile('file') file: Express.Multer.File): Promise<BlankReturnMessageDto> {
    if (!file) {
      throw new BlankReturnMessageDto(400, 'no file').toException();
    }
    const apps: AppsJson.App[] = JSON.parse(file.buffer.toString());
    return this.appService.migrateFromAppsJson(apps);
  }

  @Post('migrate/hash')
  @ApiOperation({ summary: '迁移 hash' })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  async migrateHash(@UploadedFile('file') file: Express.Multer.File): Promise<BlankReturnMessageDto> {
    return this.appService.migrateHashes();
  }

  @Put('app/:id')
  @ApiOperation({ summary: '创建 app' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async createApp(@Param('id') id: string) {
    return this.appService.createApp(id);
  }

  @Delete('app/:id')
  @ApiOperation({ summary: '删除 app' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async deleteApp(@Param('id') id: string) {
    return this.appService.deleteApp(id);
  }

  @Post('app/:id/assign')
  @ApiOperation({ summary: '设置 app 管理者' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async assignApp(@Param('id') id: string, @Body(new ValidationPipe({ transform: true })) assignAppData: AssignAppDto) {
    return this.appService.assignApp(id, assignAppData.author);
  }

  @Post('app/:id/prefix')
  @ApiOperation({ summary: '设置 app 打包前缀' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async setAppPrefix(@Param('id') id: string, @Body(new ValidationPipe({ transform: true })) appPrefix: AppPrefixDto) {
    return this.appService.setAppPrefix(id, appPrefix._prefix);
  }

  @Post('purge')
  @ApiOperation({ summary: '清理没有用的 app 包' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async purgeOldArchives() {
    return new ReturnMessageDto(201, 'success', await this.appService.purgeOldArchives());
  }
}
