import { ConsoleLogger, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import {
  ListObjectsCommand,
  PutObjectCommand,
  S3Client,
} from '@aws-sdk/client-s3';
import { createHash } from 'crypto';

@Injectable()
export class S3Service extends ConsoleLogger {
  bucket: string;
  prefix: string;
  cdnUrl: string;
  s3: S3Client;
  constructor(config: ConfigService) {
    super('s3');
    this.bucket = config.get('S3_BUCKET');
    this.prefix = config.get('S3_PREFIX');
    this.cdnUrl =
      config.get('S3_CDN_URL') ||
      `${config.get('S3_ENDPOINT')}/${this.bucket}/${this.prefix}`;
    this.s3 = new S3Client({
      credentials: {
        accessKeyId: config.get('S3_KEY'),
        secretAccessKey: config.get('S3_SECRET'),
      },
      region: config.get('S3_REGION') || 'us-west-1',
      endpoint: config.get('S3_ENDPOINT'),
      forcePathStyle: true,
    });
  }

  private async listObjects(path: string) {
    const command = new ListObjectsCommand({
      Bucket: this.bucket,
      Prefix: path,
    });
    return this.s3.send(command);
  }

  async uploadAssets(file: Express.Multer.File) {
    const fileSuffix = file.originalname.split('.').pop();
    const hash = createHash('sha512').update(file.buffer).digest('hex');
    const filename = `${hash}.${fileSuffix}`;
    const path = `${this.prefix}/${filename}`;
    const downloadUrl = `${this.cdnUrl}/${filename}`;
    const checkExisting = await this.listObjects(path);
    try {
      if (
        checkExisting.Contents &&
        checkExisting.Contents.some((obj) => obj.Key === path)
      ) {
        // already uploaded
      } else {
        await this.uploadFile(path, file.buffer, file.mimetype);
      }
      return downloadUrl;
    } catch (e) {
      this.error(`Failed to assign upload of file ${path}: ${e.toString()}`);
      return null;
    }
  }

  async uploadFile(path: string, buffer: Buffer, mime?: string) {
    return this.s3.send(
      new PutObjectCommand({
        Bucket: this.bucket,
        Key: path,
        Body: buffer,
        ContentType: mime,
      }),
    );
  }
}
