import fs from 'fs';
import path from 'path';
import child_process from 'child_process';
import os from 'os';
import util from 'util';

import _ from 'lodash';
import readdirp from 'readdirp';
import { v4 as uuidv4 } from 'uuid';

import { ConsoleLogger, Injectable } from '@nestjs/common';
import { PutObjectCommand } from '@aws-sdk/client-s3';

import { S3Service } from '../s3/s3.service';

@Injectable()
export class PackagerService extends ConsoleLogger {
  // workingPath: string;
  // releasePath: string;
  // downloadBaseUrl: string;
  // queueIdMap = new Map<string, PQueue>();

  constructor(private s3: S3Service) {
    super('packager');
  }

  async build(stream: fs.ReadStream): Promise<BuildResult> {
    const bucket_max = 10 * 1024 ** 2;
    const bucket_enter = 1 * 1024 ** 2;

    const root = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'mycard-console-'));
    await this.spawnAsync('tar', ['-zxvf', '-'], { cwd: root, stdio: [stream, 'inherit', 'inherit'] });

    const packages: Record<string, string[]> = {};
    const entries = await readdirp.promise(root, { alwaysStat: true, type: 'files_directories' });
    const [directories, files] = _.partition(entries, (item) => item.stats.isDirectory());

    // checksum
    const c = this.checksum(
      root,
      directories.map((d) => d.path),
      files.map((f) => f.path)
    );
    const promises = [c];

    // 整包
    const archive = `${uuidv4()}.tar.gz`;
    packages[archive] = [];
    promises.push(this.archive(archive, root, await fs.promises.readdir(root)));

    // 散包
    const buckets: Record<string, [string[], number]> = {};
    for (const file of files) {
      if (file.stats.size < bucket_enter) {
        const extname = path.extname(file.basename);
        buckets[extname] ??= [[], 0];
        const bucket = buckets[extname];
        if (bucket[1] + file.stats.size >= bucket_max) {
          const archive = `${uuidv4()}.tar.gz`;
          packages[archive] = bucket[0];
          promises.push(this.archive(archive, root, bucket[0]));
          bucket[0] = [];
          bucket[1] = 0;
        } else {
          bucket[0].push(file.path);
          bucket[1] += file.stats.size;
        }
      } else {
        const archive = `${uuidv4()}.tar.gz`;
        packages[archive] = [file.path];
        promises.push(this.archive(archive, root, [file.path]));
      }
    }
    for (const bucket of Object.values(buckets)) {
      if (bucket[0].length) {
        const archive = `${uuidv4()}.tar.gz`;
        packages[archive] = bucket[0];
        promises.push(this.archive(archive, root, bucket[0]));
      }
    }

    // TODO: 更新包

    const [checksum] = await Promise.all(promises); // 这个 await 过后，checksum 和 打包上传都已经跑完了
    console.log(checksum, packages);
    return { checksum, packages };
  }

  async checksum(root: string, directories: string[], files: string[]) {
    const { stdout } = await util.promisify(child_process.execFile)('sha256sum', files, { maxBuffer: 1 * 1024 ** 2 });
    return Object.fromEntries([
      ['.', ''],
      ...directories.map((d) => [d, '']),
      ...stdout.split('\n').map((line) => line.split('  ', 2).reverse()),
    ]);
  }

  archive(archive: string = `${uuidv4()}.tar.gz`, root: string, files: string[]) {
    const child = child_process.spawn('tar', ['-zcvf', '-'].concat(files), {
      cwd: root,
      stdio: ['ignore', 'pipe', 'inherit'],
    });
    return this.s3.s3.send(
      new PutObjectCommand({
        Bucket: this.s3.bucket,
        Key: archive,
        Body: child.stdout,
      })
    );
  }

  private spawnAsync(command: string, args: string[], options: child_process.SpawnOptions) {
    return new Promise<void>((resolve, reject) => {
      const child = child_process.spawn(command, args, options);
      child.on('exit', (code) => {
        if (code == 0) {
          resolve();
        } else {
          reject(code);
        }
      });
      child.on('error', (error) => {
        reject(error);
      });
    });
  }
}

export interface BuildResult {
  checksum: Record<string, string>;
  packages: Record<string, string[]>;
}
