import {
  Body,
  Controller,
  Get,
  Post,
  Query,
  UploadedFile,
  UseGuards,
  UseInterceptors,
  ValidationPipe,
} from '@nestjs/common';
import { AppService } from './app.service';
import {
  ApiBody,
  ApiConsumes,
  ApiCreatedResponse,
  ApiOkResponse,
  ApiOperation,
  ApiQuery,
} from '@nestjs/swagger';
import {
  BlankReturnMessageDto,
  GetAppReturnMessageDto,
  ReturnMessageDto,
  StringReturnMessageDto,
} from './dto/ReturnMessage.dto';
import { FetchMyCardUser, MyCardUser } from './utility/mycard-auth';
import { AppsJson } from './utility/apps-json-type';
import { MyCardAppMaintainerGuard } from './my-card-app-maintainer.guard';
import { S3Service } from './s3/s3.service';
import { FileInterceptor } from '@nestjs/platform-express';
import { FileUploadDto } from './dto/FileUpload.dto';
import AppClass = AppsJson.AppClass;

@Controller('api')
export class AppController {
  constructor(
    private readonly appService: AppService,
    private readonly s3: S3Service,
  ) {}

  @Get('apps.json')
  getAppsJson() {
    return this.appService.getAppsJson();
  }

  @Get('app')
  @ApiOperation({
    summary: '获取 app',
    description: '管理员可以查询全部的，其他用户可以查属于自己的',
  })
  @ApiQuery({ name: 'id', description: 'app 的 id，没有就是查全部' })
  @ApiOkResponse({ type: GetAppReturnMessageDto })
  getApp(@FetchMyCardUser() user: MyCardUser, @Query('id') id?: string) {
    return this.appService.getApp(user, id);
  }

  @Post('app')
  @ApiOperation({
    summary: '更新 app',
  })
  @ApiBody({ type: AppsJson.AppClass })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  updateApp(
    @FetchMyCardUser() user: MyCardUser,
    @Body(new ValidationPipe({ transform: true })) app: AppClass,
  ) {
    return this.appService.updateApp(user, app.id, app);
  }

  @Post('assets')
  @ApiOperation({
    summary: '上传附件',
    description: '必须登录用户且必须是管理员或者拥有1个 app 才能上传',
  })
  @UseInterceptors(FileInterceptor('file'))
  @ApiConsumes('multipart/form-data')
  @ApiBody({
    description: 'apps.json 文件',
    type: FileUploadDto,
  })
  @ApiCreatedResponse({ type: StringReturnMessageDto })
  @UseGuards(MyCardAppMaintainerGuard)
  async uploadAssets(@UploadedFile() file: Express.Multer.File) {
    const res = await this.s3.uploadAssets(file);
    if (res) {
      return new ReturnMessageDto(201, 'success', res);
    } else {
      throw new BlankReturnMessageDto(500, 'upload fail').toException();
    }
  }
}
