import { AfterViewInit, Component, ElementRef, OnInit, ViewChild } from '@angular/core';
import { ActivatedRoute } from '@angular/router';
import { mergeMap, tap } from 'rxjs/operators';
import { AppService, AppsJson } from './app.service';

// @ts-ignore
import { JSONEditor } from '@json-editor/json-editor';

const schema = {
  title: 'Person',
  type: 'object',
  required: ['name', 'age', 'date', 'favorite_color', 'gender', 'location', 'pets'],
  properties: {
    name: {
      type: 'string',
      description: 'First and Last name',
      minLength: 4,
      default: 'Jeremy Dorn',
    },
    age: {
      type: 'integer',
      default: 25,
      minimum: 18,
      maximum: 99,
    },
    favorite_color: {
      type: 'string',
      format: 'color',
      title: 'favorite color',
      default: '#ffa500',
    },
    gender: {
      type: 'string',
      enum: ['male', 'female', 'other'],
    },
    date: {
      type: 'string',
      format: 'date',
      options: {
        flatpickr: {},
      },
    },
    location: {
      type: 'object',
      title: 'Location',
      properties: {
        city: {
          type: 'string',
          default: 'San Francisco',
        },
        state: {
          type: 'string',
          default: 'CA',
        },
        citystate: {
          type: 'string',
          description: 'This is generated automatically from the previous two fields',
          template: '{{city}}, {{state}}',
          watch: {
            city: 'location.city',
            state: 'location.state',
          },
        },
      },
    },
    pets: {
      type: 'array',
      format: 'table',
      title: 'Pets',
      uniqueItems: true,
      items: {
        type: 'object',
        title: 'Pet',
        properties: {
          type: {
            type: 'string',
            enum: ['cat', 'dog', 'bird', 'reptile', 'other'],
            default: 'dog',
          },
          name: {
            type: 'string',
          },
        },
      },
      default: [
        {
          type: 'dog',
          name: 'Walter',
        },
      ],
    },
  },
};
JSONEditor.defaults.language = 'zh-CN';
JSONEditor.defaults.languages['zh-CN'] = {
  error_notset: 'Property must be set',
  error_notempty: 'Value required',
  error_enum: 'Value must be one of the enumerated values',
  error_const: 'Value must be the constant value',
  error_anyOf: 'Value must validate against at least one of the provided schemas',
  error_oneOf: 'Value must validate against exactly one of the provided schemas. It currently validates against {{0}} of the schemas.',
  error_not: 'Value must not validate against the provided schema',
  error_type_union: 'Value must be one of the provided types',
  error_type: 'Value must be of type {{0}}',
  error_disallow_union: 'Value must not be one of the provided disallowed types',
  error_disallow: 'Value must not be of type {{0}}',
  error_multipleOf: 'Value must be a multiple of {{0}}',
  error_maximum_excl: 'Value must be less than {{0}}',
  error_maximum_incl: 'Value must be at most {{0}}',
  error_minimum_excl: 'Value must be greater than {{0}}',
  error_minimum_incl: 'Value must be at least {{0}}',
  error_maxLength: 'Value must be at most {{0}} characters long',
  error_minLength: 'Value must be at least {{0}} characters long',
  error_pattern: 'Value must match the pattern {{0}}',
  error_additionalItems: 'No additional items allowed in this array',
  error_maxItems: 'Value must have at most {{0}} items',
  error_minItems: 'Value must have at least {{0}} items',
  error_uniqueItems: 'Array must have unique items',
  error_maxProperties: 'Object must have at most {{0}} properties',
  error_minProperties: 'Object must have at least {{0}} properties',
  error_required: "Object is missing the required property '{{0}}'",
  error_additional_properties: 'No additional properties allowed, but property {{0}} is set',
  error_property_names_exceeds_maxlength: 'Property name {{0}} exceeds maxLength',
  error_property_names_enum_mismatch: 'Property name {{0}} does not match any enum values',
  error_property_names_const_mismatch: 'Property name {{0}} does not match the const value',
  error_property_names_pattern_mismatch: 'Property name {{0}} does not match pattern',
  error_property_names_false: 'Property name {{0}} fails when propertyName is false',
  error_property_names_maxlength: 'Property name {{0}} cannot match invalid maxLength',
  error_property_names_enum: 'Property name {{0}} cannot match invalid enum',
  error_property_names_pattern: 'Property name {{0}} cannot match invalid pattern',
  error_property_names_unsupported: 'Unsupported propertyName {{0}}',
  error_dependency: 'Must have property {{0}}',
  error_date: 'Date must be in the format {{0}}',
  error_time: 'Time must be in the format {{0}}',
  error_datetime_local: 'Datetime must be in the format {{0}}',
  error_invalid_epoch: 'Date must be greater than 1 January 1970',
  error_ipv4: 'Value must be a valid IPv4 address in the form of 4 numbers between 0 and 255, separated by dots',
  error_ipv6: 'Value must be a valid IPv6 address',
  error_hostname: 'The hostname has the wrong format',
  button_save: 'Save',
  button_copy: 'Copy',
  button_cancel: 'Cancel',
  button_add: 'Add',
  button_delete_all: 'All',
  button_delete_all_title: 'Delete All',
  button_delete_last: 'Last {{0}}',
  button_delete_last_title: 'Delete Last {{0}}',
  button_add_row_title: 'Add {{0}}',
  button_move_down_title: 'Move down',
  button_move_up_title: 'Move up',
  button_properties: 'Properties',
  button_object_properties: '属性',
  button_copy_row_title: 'Copy {{0}}',
  button_delete_row_title: 'Delete {{0}}',
  button_delete_row_title_short: 'Delete',
  button_copy_row_title_short: 'Copy',
  button_collapse: '折叠',
  button_expand: '展开',
  button_edit_json: '编辑 JSON',
  button_upload: 'Upload',
  flatpickr_toggle_button: 'Toggle',
  flatpickr_clear_button: 'Clear',
  choices_placeholder_text: 'Start typing to add value',
  default_array_item_title: 'item',
  button_delete_node_warning: 'Are you sure you want to remove this node?',
};

@Component({
  selector: 'app-app',
  templateUrl: './app.component.html',
  styleUrls: ['./app.component.css'],
})
export class AppComponent implements AfterViewInit {
  app?: AppsJson.App;
  editor: JSONEditor;

  @ViewChild('form')
  form!: ElementRef<HTMLElement>;

  constructor(private route: ActivatedRoute, private appService: AppService) {}

  ngAfterViewInit(): void {
    this.route.params.pipe(mergeMap((params) => this.appService.getApp(params['id']))).subscribe((app) => {
      this.app = app;
      this.editor ??= new JSONEditor(this.form.nativeElement, {
        schema,
        theme: 'bootstrap4',
        iconlib: 'fontawesome5',
        disable_properties: true,
      });
      this.editor.setValue(this.app);
    });
  }

  submit() {
    console.log(this.editor.getValue());
  }

  reset() {
    location.reload();
  }
}
