import { Connection, IsNull, Not } from 'typeorm';
import { InjectConnection } from '@nestjs/typeorm';
import { ConsoleLogger, Injectable, Param, UploadedFile } from '@nestjs/common';
import { AppsJson } from './utility/apps-json-type';
import { App } from './entities/App.entity';
import { BlankReturnMessageDto, ReturnMessageDto } from './dto/ReturnMessage.dto';
import { FetchMyCardUser, MyCardUser } from './utility/mycard-auth';
import { PackagerService } from './packager/packager.service';

@Injectable()
export class AppService extends ConsoleLogger {
  constructor(
    @InjectConnection('app')
    private db: Connection,
    private packager: PackagerService
  ) {
    super('app');
  }

  async getAppsJson() {
    return (await this.db.getRepository(App).find({ where: { appData: Not(IsNull()), isDeleted: false } })).map((a) => a.appData);
  }

  private async updateResult<T>(f: () => Promise<T>, returnCode = 200) {
    try {
      const result = await f();
      return new ReturnMessageDto<T>(returnCode, 'success', result);
    } catch (e) {
      throw new BlankReturnMessageDto(404, 'Database Fail').toException();
    }
  }

  async migrateFromAppsJson(apps: AppsJson.App[]) {
    const targetApps: App[] = [];
    for (const appData of apps) {
      if (!appData.id) {
        throw new BlankReturnMessageDto(400, `App ${appData.name} is invalid.`).toException();
      }
      const checkExistingApp = await this.db.getRepository(App).findOne({ where: { id: appData.id }, relations: ['history'] });
      //this.error('read');
      if (checkExistingApp) {
        checkExistingApp.updateApp(appData);
        checkExistingApp.isDeleted = false;
        targetApps.push(checkExistingApp);
      } else {
        const app = new App();
        app.id = appData.id;
        app.appData = appData;
        targetApps.push(app);
      }
    }
    //this.error('write');
    return this.updateResult(async () => {
      await this.db.getRepository(App).save(targetApps);
      return;
    }, 201);
  }

  async getApp(user: MyCardUser, id?: string) {
    if (!user) {
      throw new BlankReturnMessageDto(401, 'Needs login').toException();
    }
    const query = this.db.getRepository(App).createQueryBuilder('app').where('app.isDeleted = false');
    if (!user.admin) {
      query.andWhere(':uid = ANY(app.author)', { uid: user.id });
    }
    if (id) {
      query.andWhere('app.id = :id', { id });
    }
    query.leftJoinAndSelect('app.history', 'history');
    return new ReturnMessageDto(200, 'success', await query.getMany());
  }

  async isUserCanMaintainApp(user: MyCardUser, id?: string) {
    if (user.admin) {
      return true;
    }
    const query = this.db
      .getRepository(App)
      .createQueryBuilder('app')
      .where('app.isDeleted = false')
      .andWhere(':uid = ANY(app.author)', { uid: user.id });
    if (id) {
      query.andWhere('app.id = :id', { id });
    }
    return (await query.getCount()) > 0;
  }

  async createApp(id: string) {
    let app = await this.db.getRepository(App).findOne({ where: { id }, select: ['id', 'isDeleted'] });
    if (!app) {
      app = new App();
      app.id = id;
    } else {
      if (!app.isDeleted) {
        throw new BlankReturnMessageDto(404, 'App already exists').toException();
      }
      app.isDeleted = false;
    }
    return this.updateResult(() => this.db.getRepository(App).save(app));
  }

  async assignApp(id: string, author: number[]) {
    return this.updateResult(() => this.db.getRepository(App).update({ id }, { author }), 201);
  }

  async setAppPrefix(id: string, prefix: string) {
    return this.updateResult(() => this.db.getRepository(App).update({ id }, { packagePrefix: prefix }), 201);
  }

  async updateApp(user: MyCardUser, id: string, appData: AppsJson.App) {
    if (!user) {
      throw new BlankReturnMessageDto(401, 'Needs login').toException();
    }
    appData.id = id;
    const app = await this.db.getRepository(App).findOne({
      where: { id: appData.id },
      relations: ['history'],
      select: ['id', 'author', 'appData'],
    });
    if (!app) {
      throw new BlankReturnMessageDto(404, 'App not found').toException();
    }
    if (!app.isUserCanEditApp(user)) {
      throw new BlankReturnMessageDto(403, 'Permission denied').toException();
    }
    app.updateApp(appData, user.id);
    return this.updateResult(async () => {
      await this.db.getRepository(App).save(app);
      return;
    }, 201);
  }

  async makeBuild(
    user: MyCardUser,
    file: Express.Multer.File,
    id: string,
    platform: AppsJson.Platform,
    locale: AppsJson.Locale,
    version: string
  ) {
    if (!user) {
      throw new BlankReturnMessageDto(401, 'Needs login').toException();
    }
    this.log('Build: Checking app.');
    const app = await this.db.getRepository(App).findOne({
      where: { id },
      select: ['id', 'packagePrefix', 'author'],
    });
    if (!app) {
      throw new BlankReturnMessageDto(404, 'App not found').toException();
    }
    if (!app.isUserCanEditApp(user)) {
      throw new BlankReturnMessageDto(403, 'Permission denied').toException();
    }
    const result = await this.packager.build(file.stream, app.packagePrefix);
    return new ReturnMessageDto(201, 'success', result);
  }

  async deleteApp(id: string) {
    return this.updateResult(() => this.db.getRepository(App).update({ id }, { isDeleted: true }));
  }
}
