import {
  Body,
  Controller,
  Delete,
  Get,
  Param,
  ParseIntPipe,
  Post,
  Put,
  UploadedFile,
  UseGuards,
  UseInterceptors,
  ValidationPipe,
} from '@nestjs/common';
import {
  ApiBody,
  ApiConsumes,
  ApiCreatedResponse,
  ApiOkResponse,
  ApiOperation,
  ApiTags,
} from '@nestjs/swagger';
import { MyCardAdminGuard } from '../my-card-admin.guard';
import { FileInterceptor } from '@nestjs/platform-express';
import { FileUploadDto } from '../dto/FileUpload.dto';
import { AppsJson } from '../utility/apps-json-type';
import { AppService } from '../app.service';
import { BlankReturnMessageDto } from '../dto/ReturnMessage.dto';
import { AssignAppDto } from '../dto/AssignApp.dto';

@Controller('api/admin')
@ApiTags('admin')
@UseGuards(MyCardAdminGuard)
export class AdminController {
  constructor(private readonly appService: AppService) {}

  @Post('migrate')
  @ApiOperation({ summary: '迁移', description: '上传一个 apps.json 文件' })
  @UseInterceptors(FileInterceptor('file'))
  @ApiConsumes('multipart/form-data')
  @ApiBody({
    description: 'apps.json 文件',
    type: FileUploadDto,
  })
  @ApiCreatedResponse({ type: BlankReturnMessageDto })
  async migrate(
    @UploadedFile('file') file: Express.Multer.File,
  ): Promise<BlankReturnMessageDto> {
    const apps: AppsJson.App[] = JSON.parse(file.buffer.toString());
    return this.appService.migrateFromAppsJson(apps);
  }

  @Put('app/:id')
  @ApiOperation({ summary: '创建 app' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async createApp(@Param('id') id: string) {
    return this.appService.createApp(id);
  }

  @Delete('app/:id')
  @ApiOperation({ summary: '创建 app' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async deleteApp(@Param('id') id: string) {
    return this.appService.deleteApp(id);
  }

  @Post('app/:id/assign')
  @ApiOperation({ summary: '创建 app' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async assignApp(
    @Param('id') id: string,
    @Body(new ValidationPipe({ transform: true })) assignAppData: AssignAppDto,
  ) {
    return this.appService.assignApp(id, assignAppData.author);
  }
}
