import { ConsoleLogger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import {
  _Object,
  DeleteObjectsCommand,
  HeadObjectCommand,
  ListObjectsCommand,
  PutObjectCommand,
  PutObjectCommandInput,
  S3Client,
  S3ClientConfig,
} from '@aws-sdk/client-s3';
import { createHash } from 'crypto';
import internal from 'stream';
import { Upload } from '@aws-sdk/lib-storage';

export interface S3StreamUploadResult {
  url: string;
  object: _Object;
}

export class S3Service extends ConsoleLogger {
  private getConfig(field: string) {
    return this.config.get(`${this.servicePrefix}_${field}`) || this.config.get(field);
  }

  getCdnUrl(path: string) {
    return `${this.cdnUrl}/${path}`;
  }

  private readonly bucket = this.getConfig('S3_BUCKET');
  private readonly prefix = this.getConfig('S3_PREFIX');
  public readonly cdnUrl =
    this.getConfig('S3_CDN_URL') || `${this.getConfig('S3_ENDPOINT')}/${this.bucket}${this.prefix ? `/${this.prefix}` : ''}`;
  private readonly s3Config: S3ClientConfig = {
    credentials: {
      accessKeyId: this.getConfig('S3_KEY'),
      secretAccessKey: this.getConfig('S3_SECRET'),
    },
    region: this.getConfig('S3_REGION') || 'us-west-1',
    endpoint: this.getConfig('S3_ENDPOINT'),
    forcePathStyle: true,
  };

  private readonly s3ConfigSkewed: S3ClientConfig = {
    ...this.s3Config,
    // skew 14 mins ahead for long upload
    systemClockOffset: 14 * 60 * 1000,
  };

  private readonly s3 = new S3Client(this.s3Config);
  private readonly skewedS3 = new S3Client(this.s3ConfigSkewed);

  constructor(private servicePrefix: string, private config: ConfigService) {
    super(`${servicePrefix} s3`);
  }

  async listObjects(path: string) {
    const command = new ListObjectsCommand({
      Bucket: this.bucket,
      Prefix: this.getPathWithPrefix(path),
    });
    return this.s3.send(command);
  }

  async removeObjects(paths: string[]) {
    const command = new DeleteObjectsCommand({
      Bucket: this.bucket,
      Delete: {
        Objects: paths.map((path) => ({
          Key: this.getPathWithPrefix(path),
        })),
      },
    });
    return this.s3.send(command);
  }

  async fileExists(path: string) {
    // const objects = await this.listObjects(path);
    // this.log(objects);
    // return objects.Contents ? objects.Contents.find((obj) => obj.Key === this.getPathWithPrefix(path)) : null;
    try {
      const res = await this.s3.send(
        new HeadObjectCommand({
          Bucket: this.bucket,
          Key: this.getPathWithPrefix(path),
        })
      );
      return res;
    } catch (e) {
      return;
    }
  }

  private getPathWithPrefix(filename: string) {
    return this.prefix ? `${this.prefix}/${filename}` : filename;
  }

  private getFileSuffix(filename: string) {
    const lastPattern = filename.split('/').pop();
    if (!lastPattern.length) {
      return '';
    }
    const dotSplit = lastPattern.split('.');
    if (dotSplit.length <= 1) {
      return '';
    }
    return `.${dotSplit.slice(1).join('.')}`;
  }

  async uploadAssets(file: Express.Multer.File) {
    const fileSuffix = this.getFileSuffix(file.originalname);
    const hash = createHash('sha512').update(file.buffer).digest('hex');
    const path = fileSuffix ? `${hash}${fileSuffix}` : hash;
    const downloadUrl = this.getCdnUrl(path);
    try {
      if (await this.fileExists(path)) {
        // already uploaded
        return downloadUrl;
      } else {
        return this.uploadFile(path, file.buffer, { ContentType: file.mimetype });
      }
    } catch (e) {
      this.error(`Failed to upload assets ${path}: ${e.toString()}`);
      return null;
    }
  }

  async uploadFile(path: string, content: Buffer, extras: Partial<PutObjectCommandInput> = {}) {
    await this.s3.send(
      new PutObjectCommand({
        Bucket: this.bucket,
        Key: this.getPathWithPrefix(path),
        Body: content,
        ...extras,
      })
    );
    return this.getCdnUrl(path);
  }

  async uploadStream(path: string, stream: internal.Readable, extras: Partial<PutObjectCommandInput> = {}) {
    const key = this.getPathWithPrefix(path);
    const upload = new Upload({
      client: this.skewedS3,
      params: {
        Bucket: this.bucket,
        Key: key,
        Body: stream,
        ...extras,
      },
    });
    await upload.done();
    const {
      Contents: [object],
    } = await this.listObjects(path);
    return { object, url: this.getCdnUrl(path) };
  }
}
