import { ConsoleLogger, Injectable, OnApplicationBootstrap } from '@nestjs/common';
import { InjectConnection } from '@nestjs/typeorm';
import { Connection } from 'typeorm';
import { ConfigService } from '@nestjs/config';
import { HttpService } from '@nestjs/axios';
import { lastValueFrom } from 'rxjs';
import { ReturnMessage } from '../dto/ReturnMessage.dto';
import { Archive } from '../entities/Archive.entity';
import { ArchiveMirror } from '../entities/ArchiveMirror.entity';
import { PackageS3Service } from '../package-s3/package-s3.service';
import _ from 'lodash';
import delay from 'delay';
import { LockService } from 'src/lock/lock.service';
import { createCIDR } from 'ip6addr';

export interface Middleware {
  url: string;
  identifier: string;
  maxSize?: number;
  callback?: boolean;
  singleton?: boolean;
}

export interface UploadInfo {
  url: string;
  size?: number;
  path: string;
  hash: string;
  customSuffix?: string;
  customMime?: string;
}

export interface UploadResult {
  url: string;
  middleware: Middleware;
}

@Injectable()
export class MirrorService extends ConsoleLogger implements OnApplicationBootstrap {
  private middlewares = new Map<string, Middleware>();
  private maximumMirroredSize = 0x7fffffff;
  private mirrorConcurrent = this.config.get('MIRROR_CONCURRENT') || 100;
  constructor(
    @InjectConnection('app')
    private db: Connection,
    private config: ConfigService,
    private http: HttpService,
    private packageS3: PackageS3Service,
    private redlock: LockService
  ) {
    super('mirror');
  }

  async onApplicationBootstrap() {
    const middlewareUrlStrings = this.config.get<string>('MIRROR_MIDDLEWARES');
    if (!middlewareUrlStrings) {
      return;
    }
    const middlewares = middlewareUrlStrings.split(',');
    for (const url of middlewares) {
      try {
        this.log(`Loading middleware ${url}`);
        const { data } = await lastValueFrom(this.http.get<ReturnMessage<Middleware>>(url, { responseType: 'json' }));
        const middleware: Middleware = { ...data.data, url };
        this.middlewares.set(data.data.identifier, middleware);
        if (data.data.maxSize) {
          this.maximumMirroredSize = Math.min(this.maximumMirroredSize, data.data.maxSize);
        }
        this.log(`Loaded middleware ${JSON.stringify(middleware)}`);
      } catch (e) {
        this.log(`Loading middleware ${url} failed: ${e.toString()}`);
      }
    }
    if (this.middlewares.size) {
      this.mainLoop().then();
    }
  }

  private async saveMirrorFromPath(archive: Archive) {
    if (!(await this.packageS3.fileExists(archive.archiveFullPath))) {
      this.error(`Archive ${archive.archiveFullPath} does not exist, deleting entry.`);
      await this.db.getRepository(Archive).delete({ id: archive.id });
      return [];
    }
    const uploadInfo: UploadInfo = {
      url: this.packageS3.getCdnUrl(archive.archiveFullPath),
      size: archive.size,
      path: archive.path,
      hash: archive.hash,
    };
    return _.compact(
      await Promise.all(
        Array.from(this.middlewares.values()).map(async (middleware) => {
          const result = await this.uploadWithMiddleware(uploadInfo, middleware);
          if (!result) {
            return null;
          }
          const mirrorEnt = new ArchiveMirror();
          mirrorEnt.path = archive.path;
          mirrorEnt.middleware = middleware.identifier;
          mirrorEnt.url = result.url;
          return mirrorEnt;
        })
      )
    );
  }

  private needRun = true;

  triggerMirror() {
    this.needRun = true;
  }

  private async mainLoop() {
    while (true) {
      if (!this.needRun) {
        await delay(5000);
        continue;
      }
      try {
        this.log(`Started running main loop.`);
        this.needRun = await this.runMirror();
        this.log(`Finished running main loop.`);
      } catch (e) {
        this.error(`Main loop failed: ${e.toString()}`);
        await delay(1000);
        this.needRun = true;
      }
    }
  }

  private async runMirror() {
    const query = this.db
      .createQueryBuilder()
      .select(['archive.id', 'archive.path', 'archive.size', 'archive.hash'])
      .distinctOn(['archive.path'])
      .from(Archive, 'archive')
      .where('archive.size <= :maximumMirroredSize', { maximumMirroredSize: this.maximumMirroredSize });
    query
      .andWhere(
        `not exists ${query
          .subQuery()
          .select('mirror.path')
          .from(ArchiveMirror, 'mirror')
          .where('archive.path = mirror.path')
          .andWhere('mirror.disabled = false')
          .getQuery()}`
      )
      // .orderBy('archive.id', 'DESC')
      .take(this.mirrorConcurrent);
    this.log(`Searching for archives to mirror`);
    const archives = await query.getMany();
    if (!archives.length) {
      return false;
    }
    this.log(`Uploading ${archives.length} archives.`);
    const uploadResults = _.flatten(await Promise.all(archives.map((a) => this.saveMirrorFromPath(a))));
    if (!uploadResults) {
      this.error(`Nothing uploaded, exiting.`);
      return false;
    }
    this.log(`Saving ${uploadResults.length} mirror records.`);
    await this.db.getRepository(ArchiveMirror).save(uploadResults);
    return true;
  }

  private async uploadWithMiddlewareProcess(uploadInfo: UploadInfo, middleware: Middleware): Promise<UploadResult> {
    try {
      this.log(`Uploading ${uploadInfo.url} with middleware ${middleware.identifier}.`);
      const { data } = await lastValueFrom(
        this.http.post<ReturnMessage<string>>(middleware.url, uploadInfo, { responseType: 'json', timeout: 60 * 60 * 1000 })
      );
      this.log(`Uploaded ${uploadInfo.url} to ${data.data}.`);
      return {
        url: data.data,
        middleware,
      };
    } catch (e) {
      this.error(`Failed uploading ${uploadInfo.url} with middleware ${middleware.identifier}: ${e.toString()} ${e.data}`);
      return null;
    }
  }

  private async uploadWithMiddleware(uploadInfo: UploadInfo, middleware: Middleware): Promise<UploadResult> {
    if (!middleware.singleton) {
      return this.uploadWithMiddlewareProcess(uploadInfo, middleware);
    }
    return this.redlock.using([`mirror:${middleware.identifier}`], 5000, () => this.uploadWithMiddlewareProcess(uploadInfo, middleware));
  }

  async uploadWithRandomMiddleware(uploadInfo: UploadInfo, middlewares = Array.from(this.middlewares.values())) {
    if (!middlewares.length) {
      return null;
    }
    const middleware = middlewares[Math.floor(Math.random() * middlewares.length)];
    const result = await this.uploadWithMiddleware(uploadInfo, middleware);
    if (result) {
      return result;
    }
    return this.uploadWithRandomMiddleware(
      uploadInfo,
      middlewares.filter((m) => m.identifier !== middleware.identifier)
    );
  }

  private async getMirrorUrl(mirror: ArchiveMirror, ip: string): Promise<string> {
    const middleware = this.middlewares.get(mirror.middleware);
    if (!middleware) {
      return;
    }
    if (!middleware.callback) {
      return mirror.url;
    }
    try {
      const { data } = await lastValueFrom(
        this.http.patch<ReturnMessage<string>>(middleware.url, { ...mirror, ip }, { responseType: 'json', timeout: 30000 })
      );
      return data.data;
    } catch (e) {
      this.error(`Failed getting mirror url for ${mirror.path} with middleware ${middleware.identifier}: ${e.toString()}`);
      return;
    }
  }

  private async lookForArchiveMirror(archive: Archive, ip: string) {
    if (archive.size > this.maximumMirroredSize) {
      return;
    }
    const mirrors = await this.db
      .getRepository(ArchiveMirror)
      .find({ where: { path: archive.path, disabled: false }, select: ['url', 'middleware', 'path'] });
    if (mirrors.length) {
      const urls: string[] = [];
      await Promise.all(
        mirrors.map(async (m) => {
          const url = await this.getMirrorUrl(m, ip);
          if (url) {
            urls.push(url);
          }
        })
      );
      if (urls.length) {
        archive.mirrors = urls;
      }
    }
    return archive;
  }

  mcnetworkCidrs = [
    '10.0.0.0/8',
    '192.168.0.0/16',
    '172.16.0.0/12',
    '114.112.159.128/25', // cain
  ].map((cidr) => createCIDR(cidr));

  async lookForArchivesMirror(archives: Archive[], ip: string) {
    if (this.mcnetworkCidrs.some((cidr) => cidr.contains(ip))) {
      // local network no mirror
      return archives;
    }
    //const suitableArchives = archives.filter((a) => a.size <= this.maximumMirroredSize);
    //const paths = _.uniq(suitableArchives.map((a) => a.path));
    //const mirrors = await this.db.getRepository(ArchiveMirror).find({ where: { path: archive.path, disabled: false }, select: ['url'] });
    await Promise.all(archives.map((a) => this.lookForArchiveMirror(a, ip)));
    return archives;
  }
}
