import { ConsoleLogger, Injectable } from '@nestjs/common';
import { InjectConnection } from '@nestjs/typeorm';
import { Connection } from 'typeorm';
import { ConfigService } from '@nestjs/config';
import { HttpService } from '@nestjs/axios';
import { lastValueFrom } from 'rxjs';
import { ReturnMessage } from '../dto/ReturnMessage.dto';
import { Archive } from '../entities/Archive.entity';
import { ArchiveMirror } from '../entities/ArchiveMirror.dto';
import { PackageS3Service } from '../package-s3/package-s3.service';
import _ from 'lodash';
import delay from 'delay';

export interface MiddlewareInfo {
  identifier: string;
  maxSize?: number;
}

export interface Middleware {
  url: string;
  info: MiddlewareInfo;
}

export interface UploadInfo {
  url: string;
  size?: number;
  customSuffix?: string;
  customMime?: string;
}

export interface UploadResult {
  url: string;
  middleware: Middleware;
}

@Injectable()
export class MirrorService extends ConsoleLogger {
  private middlewares: Middleware[] = [];
  private maximumMirroredSize = 0xffffffff;
  constructor(
    @InjectConnection('app')
    private db: Connection,
    private config: ConfigService,
    private http: HttpService,
    private packageS3: PackageS3Service
  ) {
    super('mirror');
    this.init().then();
  }

  private async init() {
    const middlewareUrlStrings = this.config.get<string>('MIRROR_MIDDLEWARES');
    if (!middlewareUrlStrings) {
      return;
    }
    const middlewares = middlewareUrlStrings.split(',');
    for (const url of middlewares) {
      try {
        this.log(`Loading middleware ${url}`);
        const { data } = await lastValueFrom(
          this.http.get<ReturnMessage<MiddlewareInfo>>(url, { responseType: 'json' })
        );
        const middleware: Middleware = { url, info: data.data };
        this.middlewares.push(middleware);
        if (data.data.maxSize) {
          this.maximumMirroredSize = Math.min(this.maximumMirroredSize, data.data.maxSize);
        }
        this.log(`Loaded middleware ${JSON.stringify(middleware)}`);
      } catch (e) {
        this.log(`Loading middleware ${url} failed: ${e.toString()}`);
      }
    }
    if (this.middlewares.length) {
      this.mainLoop().then();
    }
  }

  private async saveMirrorFromPath(archive: Archive) {
    if (!(await this.packageS3.fileExists(archive.archiveFullPath))) {
      return [];
    }
    const uploadInfo: UploadInfo = {
      url: this.packageS3.getCdnUrl(archive.archiveFullPath),
      size: archive.size,
    };
    return _.compact(
      await Promise.all(
        this.middlewares.map(async (middleware) => {
          const result = await this.uploadWithMiddleware(uploadInfo, middleware);
          if (!result) {
            return null;
          }
          const mirrorEnt = new ArchiveMirror();
          mirrorEnt.path = archive.path;
          mirrorEnt.middleware = middleware.info.identifier;
          mirrorEnt.url = result.url;
          return mirrorEnt;
        })
      )
    );
  }

  private needRun = true;

  triggerMirror() {
    this.needRun = true;
  }

  private async mainLoop() {
    while (true) {
      if (!this.needRun) {
        await delay(5000);
        continue;
      }
      try {
        this.log(`Started running main loop.`);
        this.needRun = await this.runMirror();
        this.log(`Finished running main loop.`);
      } catch (e) {
        this.error(`Main loop failed: ${e.toString()}`);
        await delay(1000);
        this.needRun = true;
      }
    }
  }

  private async runMirror() {
    const query = this.db
      .createQueryBuilder()
      .select(['archive.path', 'archive.size'])
      .distinctOn(['archive.path'])
      .from(Archive, 'archive')
      .where('archive.size <= :maximumMirroredSize', { maximumMirroredSize: this.maximumMirroredSize });
    query.andWhere(
      `not exists ${query.subQuery().select('mirror.path').from(ArchiveMirror, 'mirror').where('archive.path = mirror.path').getQuery()}`
    );
    //.take(100);
    this.log(`Searching for archives to mirror`);
    const archives = await query.getMany();
    if (!archives.length) {
      return false;
    }
    this.log(`Uploading ${archives.length} archives.`);
    const uploadResults = _.flatten(await Promise.all(archives.map((a) => this.saveMirrorFromPath(a))));
    if (!uploadResults) {
      this.error(`Nothing uploaded, exiting.`);
      return false;
    }
    this.log(`Saving ${uploadResults.length} mirror records.`);
    await this.db.transaction(async (edb) => {
      const entTrunks = _.chunk(uploadResults, 20000);
      for (const chunk of entTrunks) {
        await edb.getRepository(ArchiveMirror).save(chunk);
      }
    });
    return true;
  }

  private async uploadWithMiddleware(uploadInfo: UploadInfo, middleware: Middleware): Promise<UploadResult> {
    try {
      this.log(`Uploading ${uploadInfo.url} with middleware ${middleware.info.identifier}.`);
      const { data } = await lastValueFrom(
        this.http.post<ReturnMessage<string>>(middleware.url, uploadInfo, { responseType: 'json', timeout: 60 * 60 * 1000 })
      );
      this.log(`Uploaded ${uploadInfo.url} to ${data.data}.`);
      return {
        url: data.data,
        middleware,
      };
    } catch (e) {
      this.error(`Failed uploading ${uploadInfo.url} with middleware ${middleware.info.identifier}: ${e.toString()} ${e.data}`);
      return null;
    }
  }

  async uploadWithRandomMiddleware(uploadInfo: UploadInfo, middlewares = this.middlewares) {
    if (!middlewares.length) {
      return null;
    }
    const middleware = middlewares[Math.floor(Math.random() * middlewares.length)];
    const result = await this.uploadWithMiddleware(uploadInfo, middleware);
    if (result) {
      return result;
    }
    return this.uploadWithRandomMiddleware(
      uploadInfo,
      middlewares.filter((m) => m.info.identifier !== middleware.info.identifier)
    );
  }

  async lookForArchiveMirror(archive: Archive) {
    if (archive.size > this.maximumMirroredSize) {
      return;
    }
    const mirrors = await this.db.getRepository(ArchiveMirror).find({ where: { path: archive.path, disabled: false }, select: ['url'] });
    if (mirrors.length) {
      archive.mirrors = mirrors;
    }
    return archive;
  }

  async lookForArchivesMirror(archives: Archive[]) {
    //const suitableArchives = archives.filter((a) => a.size <= this.maximumMirroredSize);
    //const paths = _.uniq(suitableArchives.map((a) => a.path));
    //const mirrors = await this.db.getRepository(ArchiveMirror).find({ where: { path: archive.path, disabled: false }, select: ['url'] });
    await Promise.all(archives.map((a) => this.lookForArchiveMirror(a)));
    return archives;
  }
}
