import fs from 'fs';
import path from 'path';
import child_process from 'child_process';
import os from 'os';
import util from 'util';

import _ from 'lodash';
import { ConfigService } from '@nestjs/config';
import { PackageResult } from '../dto/PackageResult.dto';
import { PackageS3Service } from '../package-s3/package-s3.service';
import readdirp from 'readdirp';

import { ConsoleLogger, forwardRef, Inject, Injectable } from '@nestjs/common';
import { Archive, ArchiveType } from '../entities/Archive.entity';
import { AppService } from '../app.service';
import { createHash } from 'crypto';
import delay from 'delay';
import { Cache } from '@cityssm/map-expire';
import axios from 'axios';
import internal from 'stream';

export interface FileWithHash {
  file: readdirp.EntryInfo;
  hash: string;
}

export class ArchiveTask {
  readonly path: string;
  constructor(public readonly role: ArchiveType, public readonly files: FileWithHash[], public readonly altFiles?: string[]) {
    this.path = createHash('sha512')
      .update(files.map((f) => `${f.file.path}${f.hash}`).join(''))
      .digest('hex');
  }

  get archiveFullPath() {
    return `${this.path}.tar.zst`;
  }

  get filePaths() {
    return this.altFiles || this.exactFilePaths;
  }

  get exactFilePaths() {
    return this.files.map((f) => f.file.path);
  }

  get archive() {
    const archive = new Archive();
    archive.path = this.path;
    archive.role = this.role;
    archive.files = this.exactFilePaths;
    //archive.containingFiles = this.exactFilePaths.map((filePath) => ArchiveFile.fromPath(filePath));
    return archive;
  }

  addToTask(archiveTasks: ArchiveTask[]) {
    if (this.role !== ArchiveType.Part && archiveTasks.some((t) => t.path === this.path)) {
      return;
    }
    archiveTasks.push(this);
  }
}

@Injectable()
export class PackagerService extends ConsoleLogger {
  bucket_max = 10 * 1024 ** 2;
  bucket_enter = 1 * 1024 ** 2;
  packagerWorkingDirectory: string;

  private uploadLock = new Set<string>();
  private hashCache = new Cache<string, string>();

  constructor(
    @Inject(forwardRef(() => AppService)) private readonly appService: AppService,
    private s3: PackageS3Service,
    config: ConfigService
  ) {
    super('packager');
    this.bucket_max = (parseInt(config.get('PACKAGE_BUCKET_MAX')) || 10) * 1024 ** 2;
    this.bucket_enter = (parseInt(config.get('PACKAGE_BUCKET_ENTER')) || 1) * 1024 ** 2;
    this.packagerWorkingDirectory = config.get('PACKAGE_WORKING_DIRECTORY') || os.tmpdir();
  }

  private async waitForLock(key: string) {
    while (this.uploadLock.has(key)) {
      await delay(10);
    }
    this.uploadLock.add(key);
  }

  private releaseLock(key: string) {
    if (this.uploadLock.has(key)) {
      this.uploadLock.delete(key);
    }
  }

  async build(
    stream: NodeJS.ReadableStream,
    pathPrefix?: string,
    lastBuildChecksums: Record<string, string>[] = []
  ): Promise<PackageResult> {
    this.log(`Start packaging.`);
    const root = await fs.promises.mkdtemp(path.join(this.packagerWorkingDirectory, 'mycard-console-'));
    // const tarballRoot = await fs.promises.mkdtemp(path.join(this.packagerWorkingDirectory, 'mycard-console-tarball-'));
    let extractRoot = root;
    if (pathPrefix) {
      extractRoot = path.join(root, pathPrefix);
      await fs.promises.mkdir(extractRoot, { recursive: true });
    }
    try {
      this.log(`Extracting package to ${extractRoot}.`);
      stream.resume();
      //stream.on('data', (data) => this.log(`data => ${data}`));
      await this.spawnAsync('tar', ['--zstd', '-xf', '-'], { cwd: extractRoot }, stream);

      this.log(`Package extracted to ${extractRoot}.`);

      //const packagesSequence: string[][] = [];
      const entries = await readdirp.promise(root, { alwaysStat: true, type: 'files_directories' });
      const [directories, files] = _.partition(entries, (item) => item.stats.isDirectory());

      this.log(`Calculating checksums.`);

      // checksum
      const checksum = await this.checksum(
        root,
        directories.map((d) => d.path),
        files.map((f) => f.path)
      );
      const archiveTasks: ArchiveTask[] = [];

      const filesWithHash: FileWithHash[] = files.map((f) => ({ file: f, hash: checksum[f.path] }));

      // 整包
      new ArchiveTask(ArchiveType.Full, filesWithHash, await fs.promises.readdir(root)).addToTask(archiveTasks);

      // 更新包
      for (const lastChecksum of lastBuildChecksums) {
        const changedFiles = filesWithHash.filter((f) => !lastChecksum[f.file.path] || lastChecksum[f.file.path] !== f.hash);
        if (changedFiles.length) {
          new ArchiveTask(ArchiveType.Update, changedFiles).addToTask(archiveTasks);
        }
      }

      const pendingPartTasks: ArchiveTask[] = [];

      // 散包
      const buckets: Record<string, [FileWithHash[], number]> = {};
      for (const file of filesWithHash) {
        if (file.file.stats.size < this.bucket_enter) {
          const extname = path.extname(file.file.basename);
          buckets[extname] ??= [[], 0];
          const bucket = buckets[extname];
          if (bucket[1] + file.file.stats.size >= this.bucket_max) {
            new ArchiveTask(ArchiveType.Part, bucket[0]).addToTask(pendingPartTasks);
            bucket[0] = [];
            bucket[1] = 0;
          } else {
            bucket[0].push(file);
            bucket[1] += file.file.stats.size;
          }
        } else {
          new ArchiveTask(ArchiveType.Part, [file]).addToTask(pendingPartTasks);
        }
      }
      for (const bucket of Object.values(buckets)) {
        if (bucket[0].length) {
          new ArchiveTask(ArchiveType.Part, bucket[0]).addToTask(pendingPartTasks);
        }
      }

      if (pendingPartTasks.length > 1) {
        for (const task of pendingPartTasks) {
          task.addToTask(archiveTasks);
        }
      } else {
        this.log(`Skipping part archives because only 1 part archive.`);
      }

      const packages = await Promise.all(archiveTasks.map((t) => this.archive(root, t))); // 这个 await 过后，checksum 和 打包上传都已经跑完了

      //for (let i = 0; i < packagesSequence.length; ++i) {
      //  packages[gotPackages[i]] = packagesSequence[i];
      //}

      // this.log({ checksum, packages });
      this.log(`Package finished.`);
      return new PackageResult(checksum, packages);
    } catch (e) {
      throw e;
    } finally {
      await fs.promises.rm(root, { recursive: true });
      // await fs.promises.rm(tarballRoot, { recursive: true });
    }
  }

  async checksum(root: string, directories: string[], files: string[]): Promise<Record<string, string>> {
    const { stdout } = await util.promisify(child_process.execFile)('sha256sum', files, { maxBuffer: 1 * 1024 ** 3, cwd: root });
    return Object.fromEntries([
      ['.', ''],
      ...directories.map((d) => [d, '']),
      ...stdout.split('\n').map((line) => line.split('  ', 2).reverse()),
    ]);
  }

  private async lookForExistingArchiveHash(path: string) {
    let hash = this.hashCache.get(path);
    if (hash) {
      return hash;
    }
    hash = await this.appService.lookForExistingArchiveHash(path);
    this.hashCache.set(path, hash, 24 * 60 * 60 * 1000);
    return hash;
  }

  async archive(root: string, archiveTask: ArchiveTask): Promise<Archive> {
    const archive = archiveTask.archive;
    const archiveName = archiveTask.archiveFullPath;
    await this.waitForLock(archiveTask.path);
    const existing = await this.s3.fileExists(archiveName);
    if (existing) {
      archive.hash = await this.lookForExistingArchiveHash(archiveTask.path);
      archive.size = existing.Size;
      this.log(`Archive ${archiveName} exists, skipping.`);
      return archive;
    }
    const files = archiveTask.filePaths;
    this.log(`Packaging archive ${archiveName} with ${archiveTask.exactFilePaths.length} files.`);
    try {
      const child = child_process.spawn('tar', ['--zstd', '-cf', '-'].concat(files), {
        cwd: root,
      });
      const childPromise = new Promise<void>((resolve, reject) => {
        child.on('exit', (code) => {
          if (code == 0) {
            resolve();
          } else {
            reject(code);
          }
        });
        child.on('error', (error) => {
          reject(error);
        });
      });
      const hashObject = createHash('sha256');
      child.stdout.on('data', (chunk) => {
        hashObject.update(chunk);
      });
      /* if (existing) {
        await childPromise;
        archive.hash = hashObject.digest('hex');
        return archive;
      }*/
      const uploadPromise = this.s3.uploadStream(archiveName, child.stdout, {
        ContentType: 'application/tar+zstd',
      });
      const [, { object }] = await Promise.all([childPromise, uploadPromise]);
      archive.hash = hashObject.digest('hex');
      this.hashCache.set(archive.path, archive.hash, 24 * 60 * 60 * 1000);
      archive.size = object.Size;
    } catch (e) {
      throw e;
    } finally {
      this.releaseLock(archiveTask.path);
    }
    return archive;
  }

  private spawnAsync(command: string, args: string[], options: child_process.SpawnOptions, stdinStream?: NodeJS.ReadableStream) {
    return new Promise<void>((resolve, reject) => {
      const child = child_process.spawn(command, args, options);
      if (stdinStream) {
        stdinStream.pipe(child.stdin);
      }
      child.on('exit', (code) => {
        if (code == 0) {
          resolve();
        } else {
          reject(code);
        }
      });
      child.on('error', (error) => {
        reject(error);
      });
    });
  }
}
