#!/usr/bin/env node
/**
 * Created by zh99998 on 2016/12/12.
 */
import * as fs from 'fs';
import * as path from 'path';
import * as crypto from 'crypto';
import * as child_process from 'child_process';
import * as Mustache from 'mustache';
//import { sync as loadJsonFile } from 'load-json-file';
// const vercomp = require('vercomp');

const data_path = process.env.DATA_PATH || '/data/apps';
const release_path = process.env.RELEASE_PATH || '/data/release';

/*
let database;
try {
  database = loadJsonFile(path.join(data_path, 'data.json'));
} catch (error) {
  database = {};
}
 */

async function readdirRecursive(_path: string): Promise<string[]> {
  const files = await fs.promises.readdir(_path, { encoding: 'utf-8' });
  let result = files;
  for (const file of files) {
    const child = path.join(_path, file);
    const stat = await fs.promises.stat(child);
    if (stat.isDirectory()) {
      result = result.concat(
        (await readdirRecursive(child)).map((_file) => path.join(file, _file)),
      );
    }
  }
  return result;
}

function caculateSHA256(file: string): Promise<string> {
  return new Promise((resolve, reject) => {
    const input = fs.createReadStream(file);
    const hash = crypto.createHash('sha256');
    hash.on('error', (error: Error) => {
      reject(error);
    });
    input.on('error', (error: Error) => {
      reject(error);
    });
    hash.on('readable', () => {
      const data = hash.read();
      if (data) {
        resolve((<Buffer>data).toString('hex'));
      }
    });
    input.pipe(hash);
  });
}

function archive(
  archive: string,
  files: string[],
  directory: string,
): Promise<void> {
  return new Promise<void>((resolve, reject) => {
    const child = child_process.spawn(
      'tar',
      ['-zcvf', archive, '-C', directory].concat(files),
      { stdio: 'inherit' },
    );
    child.on('exit', (code) => {
      if (code == 0) {
        resolve();
      } else {
        reject(code);
      }
    });
    child.on('error', (error) => {
      reject(error);
    });
  });
}

// eslint-disable-next-line @typescript-eslint/no-empty-function
function nothing() {}

async function main(package_id) {
  console.log(`package ${package_id}`);

  await fs.promises.mkdir(release_path).catch(nothing);
  await fs.promises.mkdir(path.join(release_path, 'downloads')).catch(nothing);
  await fs.promises.mkdir(path.join(release_path, 'metalinks')).catch(nothing);
  await fs.promises.mkdir(path.join(release_path, 'checksums')).catch(nothing);
  await fs.promises.mkdir(path.join(release_path, 'dist')).catch(nothing);
  await fs.promises
    .unlink(path.join(release_path, 'downloads', `${package_id}.tar.gz`))
    .catch(nothing);
  const template = await fs.promises.readFile('template.meta4', {
    encoding: 'utf8',
  });

  // 列目录

  const package_path = path.join(data_path, package_id);

  const files = await readdirRecursive(package_path);
  files.unshift('.');

  // 计算checksum
  const checksums = new Map<string, string>();

  for (const file of files) {
    const stat = await fs.promises.stat(path.join(package_path, file));
    if (stat.isDirectory()) {
      checksums.set(file, '');
    } else {
      checksums.set(file, await caculateSHA256(path.join(package_path, file)));
    }
  }

  // 生成checksum文件
  const checksum = Array.from(checksums)
    .map(([file, checksum]) => `${checksum}  ${file}`)
    .join('\n');
  fs.promises.writeFile(
    path.join(release_path, 'checksums', package_id),
    checksum,
  );

  // 打整包
  await fs.promises
    .mkdir(path.join(release_path, 'downloads', package_id))
    .catch(nothing);
  await fs.promises
    .mkdir(path.join(release_path, 'downloads', package_id, 'full'))
    .catch(nothing);
  const archive_file = path.join(
    release_path,
    'downloads',
    package_id,
    'full',
    `${package_id}.tar.gz`,
  );

  await fs.promises.unlink(archive_file).catch(nothing);
  await archive(
    archive_file,
    await fs.promises.readdir(package_path),
    package_path,
  );

  const archive_checksum = await caculateSHA256(archive_file);
  const checksum_file = path.join(
    release_path,
    'downloads',
    package_id,
    'full',
    `${package_id}.checksum.txt`,
  );
  await fs.promises.writeFile(checksum_file, archive_checksum);
  const size_file = path.join(
    release_path,
    'downloads',
    package_id,
    'full',
    `${package_id}.size.txt`,
  );
  await fs.promises.writeFile(
    size_file,
    (await fs.promises.stat(archive_file)).size.toString(),
  );

  const link_file = path.join(
    release_path,
    'dist',
    `${archive_checksum}.tar.gz`,
  );
  await fs.promises.unlink(link_file).catch(nothing);
  await fs.promises.symlink(archive_file, link_file);

  // 整包的meta4
  const metalink = Mustache.render(template, {
    name: `${package_id}.tar.gz`,
    size: (await fs.promises.stat(archive_file)).size,
    hash: archive_checksum,
  });

  await fs.promises.writeFile(
    path.join(release_path, 'metalinks', `${package_id}.meta4`),
    metalink,
  );

  // TODO: 打近期包

  // 打散包

  await fs.promises
    .mkdir(path.join(release_path, 'downloads', package_id, 'sand'))
    .catch(nothing);
  const sand_path = path.join(release_path, 'downloads', package_id, 'sand');

  // TODO: 保留跟上一个版本相比没改动过的散包文件，无需重复打包
  for (const file of await readdirRecursive(sand_path)) {
    await fs.promises.unlink(file).catch(nothing);
  }

  for (const file of files) {
    const stat = await fs.promises.stat(path.join(package_path, file));
    if (!stat.isDirectory()) {
      const archive_file = path.join(
        release_path,
        'downloads',
        package_id,
        'sand',
        `${file.replace(/\//g, '__')}.tar.gz`,
      );
      await archive(archive_file, [file], package_path);
      const checksum_file = path.join(
        release_path,
        'downloads',
        package_id,
        'sand',
        `${file.replace(/\//g, '__')}.checksum.txt`,
      );
      const checksum = await caculateSHA256(archive_file);
      await fs.promises.writeFile(checksum_file, checksum);
      const size_file = path.join(
        release_path,
        'downloads',
        package_id,
        'sand',
        `${file.replace(/\//g, '__')}.size.txt`,
      );
      await fs.promises.writeFile(
        size_file,
        (await fs.promises.stat(archive_file)).size.toString(),
      );
      const link_file = path.join(release_path, 'dist', `${checksum}.tar.gz`);
      await fs.promises.unlink(link_file).catch(nothing);
      await fs.promises.symlink(archive_file, link_file);
    }
  }

  // TODO: 分发
}

if (process.argv[2]) {
  main(process.argv[2]);
} else {
  console.log(`param: <package>`);
}
