import { Injectable } from '@nestjs/common';
import { AppLogger } from './app.logger';
import mustache from 'mustache';
import * as fs from 'fs';
import mdToPdf from 'md-to-pdf';
import axios from 'axios';
import {
  CxcyReturnMessageDto,
  CxcyReturnMessageLike,
} from './messages/CxcyReturnMessage.dto';

export interface NameWise {
  name: string;
}

export interface ScoreSheetData {
  name: string;
  displaySex: string;
  id: string;
  collegeString: string;
  professionString: string;
  projects: NameWise[];
  theses: NameWise[];
  patents: NameWise[];
}

@Injectable()
export class AppService {
  template: string;
  cxcyUrl: string;
  constructor(private log: AppLogger) {
    this.log.setContext('app');
    this.template = fs.readFileSync('./report.md.mustache', 'utf8');
    this.cxcyUrl = process.env.CXCY_URL || 'https://cxcy.momobako.com';
  }

  async getUserData(
    userToken: string,
    requestId: string,
  ): Promise<CxcyReturnMessageLike<any>> {
    const url = `${this.cxcyUrl}/api/user/current`;
    try {
      const { data } = await axios.get(url, {
        responseType: 'json',
        params: { request_id: requestId },
        headers: { Authorization: userToken },
        validateStatus: () => true,
      });
      this.log.log(
        `Got result from ${url} with ${userToken}: ${JSON.stringify(data)}`,
      );
      return data;
    } catch (e) {
      this.log.error(`Failed to request backend ${url}: ${e.toString()}`);
      return new CxcyReturnMessageDto(
        requestId,
        500,
        'backend request fail',
        null,
      );
    }
  }

  async getPdf(data: ScoreSheetData) {
    this.log.log(`Rendering PDF with data ${JSON.stringify(data)}`);
    const renderedMarkdownContent = mustache.render(this.template, data, null, {
      escape: (v) => v,
    });
    const { content } = await mdToPdf(
      { content: renderedMarkdownContent },
      {
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        launch_options: { args: ['--no-sandbox'] },
      },
    );
    return content;
  }
}
