import {
  Body,
  Controller,
  Get,
  Post,
  Query,
  Res,
  Headers,
} from '@nestjs/common';
import { AppService, ScoreSheetData } from './app.service';
import { Response } from 'express';
import {
  CxcyReturnMessageDto,
  CxcyReturnMessageLike,
} from './messages/CxcyReturnMessage.dto';
import {
  ApiBadRequestResponse,
  ApiHeader,
  ApiInternalServerErrorResponse,
  ApiOkResponse,
  ApiOperation,
  ApiQuery,
  ApiTags,
} from '@nestjs/swagger';

@Controller('api')
@ApiTags('user')
export class AppController {
  constructor(private readonly appService: AppService) {}

  @Get('user/report.pdf')
  @ApiOperation({
    summary: '生成报告',
    description: '生成学生的创新创业电子报告。',
  })
  @ApiHeader({ name: 'Authorization', description: '用户 token' })
  @ApiQuery({ name: 'request_id', description: '请求 id，直接返回。' })
  @ApiOkResponse({ description: '成功，会直接提供文件下载。' })
  @ApiBadRequestResponse({
    description: '用户不存在或者没有登录。',
    type: CxcyReturnMessageDto,
  })
  @ApiInternalServerErrorResponse({
    description: '报告生成过程中出现问题。',
    type: CxcyReturnMessageDto,
  })
  async getPdf(
    @Headers('Authorization') userToken: string,
    @Query('request_id') requestId: string,
    @Res()
    res: Response,
  ) {
    if (!userToken) {
      new CxcyReturnMessageDto(
        requestId,
        401,
        'Empty user token.',
        null,
      ).writeResponse(res);
      return;
    }
    const data: CxcyReturnMessageLike<any> = await this.appService.getUserData(
      userToken,
      requestId,
    );
    if (!data.success) {
      res.status(data.status_code || 500).json(data);
      return;
    }
    if (!data.data || !data.data.user) {
      new CxcyReturnMessageDto(
        requestId,
        401,
        'Some contents missing.',
        null,
      ).writeResponse(res);
      return;
    }
    const userData = data.data.user;
    const pdf = await this.appService.getPdf(userData);
    res.set({
      'Content-Type': 'application/pdf',
      'Content-Disposition': encodeURIComponent(
        `attachment; filename=${userData.id}-${userData.name}.pdf`,
      ),
      'Content-Length': pdf.length,

      // prevent cache
      'Cache-Control': 'no-cache, no-store, must-revalidate',
      Pragma: 'no-cache',
      Expires: 0,
    });
    res.end(pdf);
  }
}
