import {
  CheckOutlined,
  DeleteOutlined,
  EditOutlined,
  FilterOutlined,
  SearchOutlined,
  SortAscendingOutlined,
  UndoOutlined,
} from "@ant-design/icons";
import {
  Badge,
  Button,
  ConfigProvider,
  Input,
  Space,
  type ThemeConfig,
} from "antd";
import { useEffect, useState } from "react";
import { useSnapshot } from "valtio";
import { subscribeKey } from "valtio/utils";

import { deckStore, type IDeck, initStore } from "@/stores";
import { Background, Loading, ScrollableArea, YgoCard } from "@/ui/Shared";

import { CardDetail } from "./CardDetail";
import { DeckSelect } from "./DeckSelect";
import styles from "./index.module.scss";
import { LoaderFunction } from "react-router-dom";
import { searchCards, type CardMeta } from "@/api";

const theme: ThemeConfig = {
  components: {
    Layout: {
      colorBgBody: "transparent",
    },
  },
};

export const loader: LoaderFunction = async () => {
  // 必须先加载卡组，不然页面会崩溃
  if (!initStore.decks) {
    await new Promise<void>((rs) => {
      subscribeKey(initStore, "decks", (done) => done && rs());
    });
  }
  return null;
};

export const Component: React.FC = () => {
  const snapDecks = useSnapshot(deckStore);
  const { sqlite } = useSnapshot(initStore);
  const [selectedDeck, setSelectedDeck] = useState<IDeck>(deckStore.decks[0]);

  return (
    <ConfigProvider theme={theme}>
      <Background />
      <div className={styles.layout} style={{ width: "100%" }}>
        <div className={styles.sider}>
          <ScrollableArea className={styles["deck-select-container"]}>
            <DeckSelect
              decks={snapDecks.decks}
              selected={selectedDeck.deckName}
              onSelect={(name) =>
                setSelectedDeck(deckStore.get(name) ?? deckStore.decks[0])
              }
              onDelete={(id) => console.log(id)}
              onDownload={(id) => console.log(id)}
              onAdd={() => console.log("add")}
            />
          </ScrollableArea>
          <CardDetail code={123} open={false} onClose={() => {}} />
        </div>
        <div className={styles.content}>
          <div className={styles.deck}>
            <DeckEditor deck={selectedDeck} onSave={() => {}} />
          </div>
          <div className={styles.select}>
            {sqlite.progress === 1 ? (
              <CardSelect />
            ) : (
              <div className={styles.container}>
                <Loading />
              </div>
            )}
          </div>
        </div>
      </div>
    </ConfigProvider>
  );
};
Component.displayName = "Build";

/** 正在编辑的卡组 */
const DeckEditor: React.FC<{
  deck: IDeck;
  onSave: (deck: IDeck) => void;
}> = ({ deck, onSave }) => {
  const [editingDeck, setEditingDeck] = useState<IDeck>(deck);
  useEffect(() => {
    setEditingDeck(deck);
  }, [deck]);
  return (
    <div className={styles.container}>
      <Space className={styles.title}>
        <Input
          placeholder="请输入卡组名字"
          bordered={false}
          prefix={<EditOutlined />}
          style={{ width: 400 }}
          onChange={(e) =>
            setEditingDeck({ ...editingDeck, deckName: e.target.value })
          }
          value={editingDeck.deckName}
        />
        <Space style={{ marginRight: 6 }}>
          <Button type="text" size="small" icon={<DeleteOutlined />}>
            清空
          </Button>
          <Button type="text" size="small" icon={<UndoOutlined />}>
            重置
          </Button>
          <Button
            type="text"
            size="small"
            icon={<CheckOutlined />}
            onClick={() => onSave(editingDeck)}
          >
            保存
          </Button>
        </Space>
      </Space>
      <ScrollableArea className={styles["deck-zone"]}>
        {(["main", "extra", "side"] as const).map((type) => (
          <div key={type} className={styles[type]}>
            <div className={styles["card-continer"]}>
              {editingDeck[type].map((code, i) => (
                <div className={styles.card} key={i}>
                  <YgoCard code={code} />
                </div>
              ))}
            </div>
          </div>
        ))}
      </ScrollableArea>
    </div>
  );
};

/** 卡片库，选择卡片加入正在编辑的卡组 */
const CardSelect: React.FC = () => {
  const [searchWord, setSearchWord] = useState("");
  const [searchResult, setSearchResult] = useState<CardMeta[]>([]);
  const handleSearch = async () => {
    const result = await searchCards(searchWord);
    setSearchResult(result);
  };
  return (
    <div className={styles.container}>
      <div className={styles.title}>
        <Input
          placeholder="搜索卡片"
          bordered={false}
          suffix={
            <Button
              type="text"
              icon={<SearchOutlined />}
              onClick={handleSearch}
            />
          }
          value={searchWord}
          onChange={(e) => setSearchWord(e.target.value)}
          onKeyUp={(e) => e.key === "Enter" && handleSearch()}
        />
      </div>
      <div className={styles["select-btns"]}>
        <Button block type="text" icon={<FilterOutlined />}>
          筛选
          {/* TODO: 下面这个Badge应根据有无筛选规则而显示 */}
          {false && <Badge dot offset={[5, -5]} />}
        </Button>
        <Button block type="text" icon={<SortAscendingOutlined />}>
          <span>
            排列
            <span className={styles["search-count"]}>
              ({searchResult.length})
            </span>
          </span>
          {false && <Badge dot offset={[5, -5]} />}
        </Button>
        <Button block type="text" icon={<DeleteOutlined />}>
          重置
        </Button>
      </div>
      <ScrollableArea className={styles["search-cards-container"]}>
        <div className={styles["search-cards"]}>
          {searchResult.map(({ id }, i) => (
            <div className={styles.card} key={i}>
              <YgoCard code={id} />
            </div>
          ))}
        </div>
      </ScrollableArea>
    </div>
  );
};
