use super::{cards_service, CardMeta};
use std::{
    collections::HashMap,
    fs::File,
    io::{BufRead, BufReader},
    path::Path,
};

type StringsManager = HashMap<String, HashMap<i64, String>>;

// TODO: `strings_manager`应该做好持久化存储
pub fn service_maker(strings_manager: StringsManager) -> impl Fn(String) -> String + Clone {
    move |param| {
        // TODO：应该做好错误处理
        let (r#type, code) = param.split_once('_').unwrap_or_default();
        let code = code.parse::<i64>().unwrap_or_default();

        if r#type == "!card" {
            let card_code = (code >> 4) & 0x0fffffff;
            let offset = code & 0xf;

            let card =
                serde_json::from_str::<CardMeta>(cards_service(card_code.to_string()).as_str())
                    .unwrap_or_default();

            card.text
                .get_str_by_offset(offset as usize)
                .unwrap_or_default()
        } else {
            strings_manager
                .get(r#type)
                .map_or(Some("".to_string()), |hash| hash.get(&code).cloned())
                .unwrap_or_default()
        }
    }
}

pub fn strings_conf_reader(path: impl AsRef<Path>) -> anyhow::Result<StringsManager> {
    let f = File::open(path)?;
    let reader = BufReader::new(f);

    let mut results: StringsManager = HashMap::new();
    for line in reader.lines() {
        let line = line?;

        if !line.starts_with('#') {
            if let Some((r#type, left)) = line.split_once(' ') {
                if let Some((code, s)) = left.split_once(' ') {
                    results
                        .entry(r#type.to_string())
                        .or_default()
                        .insert(parse_code(code)?, s.to_string());
                }
            }
        }
    }
    Ok(results)
}

fn parse_code(code: &str) -> anyhow::Result<i64> {
    if code.starts_with("0x") {
        let without_prefix = code.trim_start_matches("0x");
        Ok(i64::from_str_radix(without_prefix, 16)?)
    } else {
        Ok(code.parse()?)
    }
}

#[cfg(test)]
mod tests {
    use super::strings_conf_reader;

    #[test]
    fn test_strings_conf_reader() {
        let workspace = env!("CARGO_MANIFEST_DIR");
        let strings_manager = strings_conf_reader(format!(
            "{workspace}/ygopro-database/locales/zh-CN/strings.conf"
        ))
        .unwrap_or_default();

        assert_eq!(
            strings_manager.get("!system").unwrap().get(&100).unwrap(),
            "先攻"
        );
        assert_eq!(
            strings_manager.get("!counter").unwrap().get(&0x56).unwrap(),
            "炎星指示物"
        );
        assert_eq!(
            strings_manager
                .get("!setname")
                .unwrap()
                .get(&0x3008)
                .unwrap(),
            "元素英雄	E・HERO"
        );
    }

    #[test]
    fn test_card_strings_reader() {
        let data = 32807846;
        let code = (data >> 4) & 0xfffffff;
        let offset = data & 0xf;

        println!("{}, {}", code, offset);

        let card = serde_json::from_str::<super::CardMeta>(
            super::cards_service(code.to_string()).as_str(),
        )
        .unwrap();

        assert_eq!(card.text.get_str_by_offset(offset).unwrap(), "");
    }
}
