mod model;
mod schema;
use crate::infra::DbConn;
use diesel::{ExpressionMethods, QueryDsl, RunQueryDsl};
use std::collections::HashMap;

pub use model::{CardDatas, CardTexts};

/// `cards.cdb`业务层模块
pub struct CardsDB {
    conn: DbConn,
}

impl CardsDB {
    /// 创建`CardsDB`实例
    ///
    /// TODO: 这里更好得做法应该是由基建层维护db连接池，
    /// 业务方通过取到db连接来做业务逻辑
    pub fn new() -> anyhow::Result<Self> {
        let workspace = env!("CARGO_MANIFEST_DIR");
        let db_url = format!("{workspace}/ygopro-database/locales/zh-CN/cards.cdb");
        let conn = DbConn::establish_sqlite(&db_url)?;

        Ok(Self { conn })
    }

    /// 通过`id`获取`Card`元数据
    pub fn select_card_datas(&mut self, ids: &[i64]) -> anyhow::Result<HashMap<i64, CardDatas>> {
        use schema::datas;

        let conn = &mut *self.conn;
        let records: Vec<CardDatas> = datas::table.filter(datas::id.eq_any(ids)).load(conn)?;

        let records = records
            .into_iter()
            .map(|record| (record.id, record))
            .collect();
        Ok(records)
    }

    /// 通过`id`获取`Card`文本数据
    pub fn select_card_texts(&mut self, ids: &[i64]) -> anyhow::Result<HashMap<i64, CardTexts>> {
        use schema::texts;

        let conn = &mut *self.conn;
        let records: Vec<CardTexts> = texts::table.filter(texts::id.eq_any(ids)).load(conn)?;

        let records = records
            .into_iter()
            .map(|record| (record.id, record))
            .collect();

        Ok(records)
    }
}

#[cfg(test)]
mod tests {
    use super::{
        model::{CardDatas, CardTexts},
        CardsDB,
    };

    #[test]
    fn test_select_card_datas() {
        let mut db = CardsDB::new().unwrap();

        let id = 2333365;

        let data = db.select_card_datas(&[id]).unwrap().remove(&id).unwrap();

        assert_eq!(
            data,
            CardDatas {
                id: 2333365,
                ot: 3,
                alias: 0,
                setcode: 66,
                type_: 33,
                atk: 2000,
                def: 2000,
                level: 4,
                race: 1,
                attribute: 16,
                category: 65536
            }
        )
    }

    #[test]
    fn test_select_card_texts() {
        let mut db = CardsDB::new().unwrap();

        let id = 2333365;

        let text = db.select_card_texts(&[id]).unwrap().remove(&id).unwrap();

        assert_eq!(
            text,
            CardTexts {
                id: 2333365,
                name: "极星将 提尔".into(),
                desc: Some(
                    "场上没有这张卡以外的名字带有「极星」的怪兽表侧表示存在的场合，这张卡破坏。\
                     只要这张卡在场上表侧表示存在，对方不能选择「极星将 \
                     提尔」以外的名字带有「极星」的怪兽作为攻击对象。"
                        .into()
                ),
                str1: Some("".into()),
                str2: Some("".into()),
                str3: Some("".into()),
                str4: Some("".into()),
                str5: Some("".into()),
                str6: Some("".into()),
                str7: Some("".into()),
                str8: Some("".into()),
                str9: Some("".into()),
                str10: Some("".into()),
                str11: Some("".into()),
                str12: Some("".into()),
                str13: Some("".into()),
                str14: Some("".into()),
                str15: Some("".into()),
                str16: Some("".into())
            }
        )
    }
}
