import os
import sys
import urllib.request
from datetime import datetime, timedelta
from typing import Dict, Optional
from pathlib import Path

# 配置
CONFIG = {
    'folder_path': "deck",
    'formal_file': "to-formal.txt",
    'pre_file': "to-pre.txt",
    'url_formal': "https://code.moenext.com/coccvo/card-password-conversion/-/raw/master/to-formal.txt",
    'url_pre': "https://code.moenext.com/coccvo/card-password-conversion/-/raw/master/to-pre.txt"
}

def check_and_update_file(filename: str, url: str) -> bool:
    """检查并更新文件，如果成功返回True，失败返回False"""
    try:
        if not os.path.exists(filename):
            print(f"{filename} 不存在，开始下载...")
            return download_file(url, filename)
        
        file_mod_time = datetime.fromtimestamp(os.path.getmtime(filename))
        current_time = datetime.now()
        if (current_time - file_mod_time) > timedelta(days=1):
            print(f"更新 {filename}...")
            return download_file(url, filename)
        return True
    except Exception as e:
        print(f"Error: 检查文件更新失败 {e}")
        return False
    
def download_file(url: str, filename: str) -> bool:
    try:
        urllib.request.urlretrieve(url, filename)
        return True
    except Exception as e:
        print(f"Error: 下载对照表失败 {e}")
        return False


def load_replacements(formal_file: str, pre_file: str) -> Optional[Dict[str, str]]:
    try:
        replacements = {}
        # 读取正式对照表
        with open(formal_file, 'r', encoding='utf-8') as file:
            for line in file:
                parts = line.strip().split('\t')
                if len(parts) == 2:
                    replacements[parts[0]] = parts[1]

        # 读取预发布对照表
        with open(pre_file, 'r', encoding='utf-8') as file:
            for line in file:
                parts = line.strip().split('\t')
                if len(parts) == 2:
                    replacements[parts[1]] = parts[0]
        
        return replacements
    except Exception as e:
        print(f"Error: 读取对照表失败 {e}")
        return None

def main() -> int:
    # 检查目录
    if not os.path.exists(CONFIG['folder_path']):
        print("Error: 请把此文件放到游戏目录中再运行")
        return 1

    # 更新对照表
    for file_key, url_key in [('formal_file', 'url_formal'), ('pre_file', 'url_pre')]:
        if not check_and_update_file(CONFIG[file_key], CONFIG[url_key]):
            return 1

    # 加载替换规则
    replacements = load_replacements(CONFIG['formal_file'], CONFIG['pre_file'])
    if not replacements:
        return 1

    # 处理卡组文件
    deck_path = Path(CONFIG['folder_path'])
    for ydk_file in deck_path.rglob('*.ydk'):
        try:
            content = ydk_file.read_text(encoding='utf-8')
            new_content = content
            for old, new in replacements.items():
                new_content = new_content.replace(old, new)

            if new_content != content:
                ydk_file.write_text(new_content, encoding='utf-8')
                print(f"更新卡组: {ydk_file}")
            else:
                print(f"卡组无需修改: {ydk_file}")
        except Exception as e:
            print(f"处理文件 {ydk_file} 时出错: {e}")
            continue

    print("卡组更新完成。")
    return 0

if __name__ == "__main__":
    try:
        exit_code = main()
        print("\n按回车键退出...")
        input()
        sys.exit(exit_code)  # 使用 sys.exit() 替代 exit()
    except Exception as e:
        print(f"Error: {e}")
        print("\n按回车键退出...")
        input()
        sys.exit(1)
