import os
import sys
import urllib.request
import urllib.error
from typing import Optional, List
from pathlib import Path

# 配置
CONFIG = {
    'folder_path': "deck",
    'formal_file': "to-formal.txt",
    'pre_file': "to-pre.txt",
    'url_formal': "https://code.moenext.com/coccvo/card-password-conversion/-/raw/master/to-formal.txt",
    'url_pre': "https://code.moenext.com/coccvo/card-password-conversion/-/raw/master/to-pre.txt",
    'max_retries': 3
}

def get_etag_path(filename: str) -> str:
    """获取存储 ETag 的隐藏文件路径"""
    return f"{filename}.etag"

def load_local_etag(filename: str) -> Optional[str]:
    """从本地加载 ETag"""
    etag_file = get_etag_path(filename)
    if os.path.exists(etag_file):
        try:
            with open(etag_file, 'r', encoding='utf-8') as f:
                return f.read().strip()
        except Exception:
            pass
    return None

def save_local_etag(filename: str, etag: str):
    """保存 ETag 到本地"""
    etag_file = get_etag_path(filename)
    try:
        with open(etag_file, 'w', encoding='utf-8') as f:
            f.write(etag)
    except Exception as e:
        print(f"警告：无法保存 ETag 文件 {e}")

def check_and_update_file(filename: str, url: str) -> bool:
    """
    检查并更新文件 (使用 ETag 机制，无随机延迟)
    """
    local_etag = load_local_etag(filename)
    
    for attempt in range(CONFIG['max_retries']):
        try:
            req = urllib.request.Request(url)
            req.add_header('User-Agent', 'YDK-Update-Tool/1.0')
            
            # 核心：如果有本地 ETag，添加到请求头
            if local_etag:
                req.add_header('If-None-Match', local_etag)
            
            try:
                with urllib.request.urlopen(req, timeout=10) as response:
                    # 状态码 200: 文件有更新或首次下载
                    content = response.read()
                    new_etag = response.headers.get('ETag', '').strip()
                    
                    # 保存文件内容
                    with open(filename, 'wb') as f:
                        f.write(content)
                    
                    # 保存新的 ETag
                    if new_etag:
                        save_local_etag(filename, new_etag)
                    
                    print(f"[下载] {filename} 已更新。")
                    return True

            except urllib.error.HTTPError as e:
                # 处理 304 Not Modified (文件未变化)
                if e.code == 304:
                    return True
                else:
                    # 其他 HTTP 错误
                    raise e

        except Exception as e:
            wait_time = (attempt + 1) * 2
            print(f"[警告] 下载 {filename} 失败 (尝试 {attempt+1}/{CONFIG['max_retries']}): {e}")
            
            if attempt < CONFIG['max_retries'] - 1:
                print(f"      等待 {wait_time} 秒后重试...")
                import time
                time.sleep(wait_time)
            else:
                if os.path.exists(filename):
                    print(f"[提示] 无法连接服务器，继续使用本地旧版 {filename}。")
                    return True
                else:
                    print(f"[错误] 无法下载初始文件 {filename}。")
                    return False
    
    return False

def load_replacements(formal_file: str, pre_file: str) -> Optional[dict]:
    replacements = {}
    try:
        if not os.path.exists(formal_file):
            return None
            
        with open(formal_file, 'r', encoding='utf-8') as f:
            for line in f:
                parts = line.strip().split('\t')
                if len(parts) == 2:
                    replacements[parts[0]] = parts[1]

        if os.path.exists(pre_file):
            with open(pre_file, 'r', encoding='utf-8') as f:
                for line in f:
                    parts = line.strip().split('\t')
                    if len(parts) == 2:
                        replacements[parts[1]] = parts[0]
        return replacements
    except Exception as e:
        print(f"[错误] 读取对照表出错：{e}")
        return None

def main():
    # 1. 检查目录
    if not os.path.exists(CONFIG['folder_path']):
        print(f"[错误] 未找到 '{CONFIG['folder_path']}' 文件夹。")
        print("       请将此脚本放在游戏目录下运行。")
        input("\n按回车键退出...")
        return 1

    # 2. 更新对照表
    files_config = [
        ('formal_file', 'url_formal'),
        ('pre_file', 'url_pre')
    ]

    critical_failure = False
    for file_key, url_key in files_config:
        f_name = CONFIG[file_key]
        u_url = CONFIG[url_key]
        
        if not os.path.exists(f_name):
            if not check_and_update_file(f_name, u_url):
                critical_failure = True
        else:
            check_and_update_file(f_name, u_url)

    if critical_failure:
        print("\n[错误] 缺少必要的对照表文件且无法下载。")
        input("\n按回车键退出...")
        return 1

    # 3. 加载替换规则
    replacements = load_replacements(CONFIG['formal_file'], CONFIG['pre_file'])
    
    if not replacements:
        print("[警告] 替换规则为空，跳过卡组处理。")
    else:
        deck_path = Path(CONFIG['folder_path'])
        ydk_files = list(deck_path.rglob('*.ydk'))
        
        if not ydk_files:
            print(f"[提示] 在 '{CONFIG['folder_path']}' 目录下未找到 .ydk 文件。")
        else:
            updated_files: List[str] = []
            
            for ydk in ydk_files:
                try:
                    content = ydk.read_text(encoding='utf-8')
                    new_content = content
                    
                    # 执行替换
                    for old, new in replacements.items():
                        if old in new_content:
                            new_content = new_content.replace(old, new)
                    
                    # 如果内容发生变化，写入文件并记录
                    if new_content != content:
                        ydk.write_text(new_content, encoding='utf-8')
                        try:
                            relative_path = ydk.relative_to(deck_path)
                            updated_files.append(str(relative_path))
                        except ValueError:
                            updated_files.append(str(ydk.name))
                
                except Exception as e:
                    print(f"[错误] 处理文件 {ydk.name} 时异常：{e}")

            # 4. 显示修改列表
            print("\n" + "="*30)
            if updated_files:
                print(f"【更新完成】共修改 {len(updated_files)} 个卡组文件：")
                for fname in updated_files:
                    print(f"  - {fname}")
            else:
                print("【更新完成】所有卡组文件均为最新，无需修改。")
            print("="*30)

    print("\n按回车键退出...")
    input()
    return 0

if __name__ == "__main__":
    try:
        exit_code = main()
        sys.exit(exit_code)
    except KeyboardInterrupt:
        print("\n[中断] 用户取消操作。")
        sys.exit(130)
    except Exception as e:
        print(f"\n[严重错误] {e}")
        input("\n按回车键退出...")
        sys.exit(1)
