import { Database } from "sql.js";

import { CardData, CardMeta, CardText } from "@/api";

interface FtsConditions {
  // 过滤条件
  types?: number[]; // 卡片类型
  levels?: number[]; // 星阶/刻度/link值
  atk?: [number, number]; // 攻击力区间
  def?: [number, number]; // 防御力区间
  race?: number; // 种族
  attribute?: number; // 属性
}
export interface FtsParams {
  query: string; // 用于全文检索的query
  conditions: FtsConditions; // 过滤条件
}

export function invokeFts(db: Database, params: FtsParams): CardMeta[] {
  const { query, conditions } = params;
  const ftsTexts: CardText[] = [];
  const ftsMetas: CardMeta[] = [];

  const textStmt = db.prepare("SELECT * FROM texts WHERE name LIKE $query");
  textStmt.bind({ $query: `%${query}%` });

  while (textStmt.step()) {
    const row = textStmt.getAsObject();
    ftsTexts.push(row);
  }

  for (const text of ftsTexts) {
    const id = text.id;
    if (id) {
      const sql = `SELECT * FROM datas ${getFtsCondtions(conditions)}`;
      const dataStmt = db.prepare(sql);
      const data: CardData = dataStmt.getAsObject({ $id: id });

      ftsMetas.push({ id, data, text });
    }
  }

  return ftsMetas;
}

function getFtsCondtions(conditions: FtsConditions): string {
  const { types, levels, atk, def, race, attribute } = conditions;

  const typesCondition = types
    ?.map((type) => `(types & ${type}) > 0`)
    .join(" OR ");
  const levelsCondition = levels
    ?.map((level) => `level = ${level}`)
    .join(" OR ");
  const atkCondition = atk ? `atk BETWEEN ${atk[0]} AND ${atk[1]}` : undefined;
  const defCondition = def ? `def BETWEEN ${def[0]} AND ${def[1]}` : undefined;
  const raceCondition = race !== undefined ? `race = ${race}` : undefined;
  const attributeCondition =
    attribute !== undefined ? `attribute = ${attribute}` : undefined;

  const merged = [
    typesCondition,
    levelsCondition,
    atkCondition,
    defCondition,
    raceCondition,
    attributeCondition,
  ]
    .filter((condition) => condition !== undefined)
    .map((condition) => `(${condition})`)
    .join(" AND ");

  return merged !== "" ? `WHERE ${merged}` : "";
}
