import { Button, Input, Modal } from "antd";
import React, { ChangeEvent, useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { proxy, useSnapshot } from "valtio";

import { useConfig } from "@/config";
import { accountStore } from "@/stores";

import styles from "./Modal.module.scss";
const NeosConfig = useConfig();
const serverConfig = NeosConfig.servers;

const {
  defaults: { defaultPlayer, defaultPassword },
  automation: { isAiMode },
} = useConfig();

interface Props {
  open?: boolean;
}

const defaultProps: Props = {
  open: false,
};

export const localStore = proxy<Props>(defaultProps);

export const MatchModal: React.FC = ({}) => {
  const { open } = useSnapshot(localStore);
  const { user } = useSnapshot(accountStore);
  const [player, setPlayer] = useState(user?.name ?? defaultPlayer);
  const [passwd, setPasswd] = useState(defaultPassword);
  const [server, setServer] = useState(
    `${serverConfig[0].ip}:${serverConfig[0].port}`
  );
  const [confirmLoading, setConfirmLoading] = useState(false);
  const navigate = useNavigate();

  let handlePlayerChange = (event: ChangeEvent<HTMLInputElement>) => {
    setPlayer(event.target.value);
  };
  let handleServerChange = (event: ChangeEvent<HTMLInputElement>) => {
    setServer(event.target.value);
  };
  let handlePasswdChange = (event: ChangeEvent<HTMLInputElement>) => {
    setPasswd(event.target.value);
  };

  // 因为萌卡服务器的房间密码会有`#`等特殊字符，因此这里用`encodeURIComponent`做下转义
  const handleSubmit = () => {
    setConfirmLoading(true);
    // TODO: 这里应该进行WASM，websocket等的初始化，成功加入房间后才跳转
    // navigate(`/room/${player}/${encodeURIComponent(passwd)}/${server}`);
  };

  useEffect(() => {
    // 如果开启了AI模式，直接进入房间
    if (isAiMode) {
      handleSubmit();
    }
  }, []);

  return (
    <Modal
      open={open}
      title="请输入自定义房间信息"
      onCancel={() => (localStore.open = false)}
      footer={
        <Button onClick={handleSubmit} loading={confirmLoading}>
          加入房间
        </Button>
      }
      confirmLoading={confirmLoading}
      centered
    >
      <div className={styles["inputs-container"]}>
        <Input
          className={styles.input}
          type="text"
          placeholder="玩家昵称"
          value={player}
          onChange={handlePlayerChange}
          required
        />
        <Input
          className={styles.input}
          type="text"
          placeholder="服务器(IP+端口)"
          value={server}
          onChange={handleServerChange}
          required
        />
        <Input
          className={styles.input}
          type="password"
          autoCorrect="off"
          placeholder="房间密码(可选)"
          value={passwd}
          onChange={handlePasswdChange}
        />
      </div>
    </Modal>
  );
};
