import {
  CheckOutlined,
  DeleteOutlined,
  EditOutlined,
  FilterOutlined,
  SearchOutlined,
  SortAscendingOutlined,
  UndoOutlined,
} from "@ant-design/icons";
import {
  Badge,
  Button,
  ConfigProvider,
  Input,
  Space,
  type ThemeConfig,
} from "antd";
import { useEffect, useState } from "react";
import { useSnapshot } from "valtio";

import { deckStore, type IDeck } from "@/stores";
import { Background, ScrollableArea, YgoCard } from "@/ui/Shared";

import { CardDetail } from "./CardDetail";
import { DeckSelect } from "./DeckSelect";
import styles from "./index.module.scss";

const theme: ThemeConfig = {
  components: {
    Layout: {
      colorBgBody: "transparent",
    },
  },
};

export const Component: React.FC = () => {
  const snapDecks = useSnapshot(deckStore);
  const [selectedDeck, setSelectedDeck] = useState<IDeck>(deckStore.decks[0]);

  return (
    <ConfigProvider theme={theme}>
      <Background />
      <div className={styles.layout} style={{ width: "100%" }}>
        <div className={styles.sider}>
          <ScrollableArea className={styles["deck-select-container"]}>
            <DeckSelect
              decks={snapDecks.decks}
              selected={selectedDeck.deckName}
              onSelect={(name) =>
                setSelectedDeck(deckStore.get(name) ?? deckStore.decks[0])
              }
              onDelete={(id) => console.log(id)}
              onDownload={(id) => console.log(id)}
              onAdd={() => console.log("add")}
            />
          </ScrollableArea>
          <CardDetail code={123} open={false} onClose={() => {}} />
        </div>
        <div className={styles.content}>
          <Deck deck={selectedDeck} onSave={() => {}} />
          <CardSelect />
        </div>
      </div>
    </ConfigProvider>
  );
};
Component.displayName = "Build";

/** 正在编辑的卡组 */
const Deck: React.FC<{
  deck: IDeck;
  onSave: (deck: IDeck) => void;
}> = ({ deck, onSave }) => {
  const [editingDeck, setEditingDeck] = useState<IDeck>(deck);
  useEffect(() => {
    setEditingDeck(deck);
  }, [deck]);
  return (
    <div className={styles.deck}>
      <div className={styles.container}>
        <Space className={styles.title}>
          <Input
            placeholder="请输入卡组名字"
            bordered={false}
            prefix={<EditOutlined />}
            style={{ width: 400 }}
            onChange={(e) =>
              setEditingDeck({ ...editingDeck, deckName: e.target.value })
            }
            value={editingDeck.deckName}
          />
          <Space style={{ marginRight: 6 }}>
            <Button type="text" size="small" icon={<DeleteOutlined />}>
              清空
            </Button>
            <Button type="text" size="small" icon={<UndoOutlined />}>
              重置
            </Button>
            <Button
              type="text"
              size="small"
              icon={<CheckOutlined />}
              onClick={() => onSave(editingDeck)}
            >
              保存
            </Button>
          </Space>
        </Space>
        <ScrollableArea className={styles["deck-zone"]}>
          {(["main", "extra", "side"] as const).map((type) => (
            <div className={styles[type]}>
              <div className={styles["card-continer"]}>
                {editingDeck[type].map((code, i) => (
                  <div className={styles.card} key={i}>
                    <YgoCard code={code} />
                  </div>
                ))}
              </div>
            </div>
          ))}
        </ScrollableArea>
      </div>
    </div>
  );
};

/** 卡片库，选择卡片加入正在编辑的卡组 */
const CardSelect: React.FC = () => {
  return (
    <div className={styles.select}>
      <div className={styles.container}>
        <div className={styles.title}>
          <Input
            placeholder="搜索卡片"
            bordered={false}
            suffix={<Button type="text" icon={<SearchOutlined />} />}
          />
        </div>
        <div className={styles["select-btns"]}>
          <Button block type="text" icon={<FilterOutlined />}>
            筛选
            {/* TODO: 下面这个Badge应根据有无筛选规则而显示 */}
            {false && <Badge dot offset={[5, -5]} />}
          </Button>
          <Button block type="text" icon={<SortAscendingOutlined />}>
            排列
            {false && <Badge dot offset={[5, -5]} />}
          </Button>
          <Button block type="text" icon={<DeleteOutlined />}>
            重置
          </Button>
        </div>
        <ScrollableArea className={styles["search-cards-container"]}>
          <div className={styles["search-cards"]}>
            {Array.from({ length: 60 }).map((_, i) => (
              <div className={styles.card} key={i} />
            ))}
          </div>
        </ScrollableArea>
      </div>
    </div>
  );
};
