import { EditFilled, SettingFilled } from "@ant-design/icons";
import { ConfigProvider, Select, Space } from "antd";
import classNames from "classnames";
import { type LoaderFunction } from "react-router-dom";

import { CookieKeys, setCookie } from "@/api";
import { accountStore, type User } from "@/stores";
import { Background, IconFont } from "@/ui/Shared";

import styles from "./index.module.scss";

export const loader: LoaderFunction = () => {
  const sso = new URLSearchParams(location.search).get("sso");
  const user = sso ? getSSOUser(new URLSearchParams(atob(sso))) : undefined;
  if (user) {
    accountStore.login(user);
    setCookie(CookieKeys.USER, JSON.stringify(user));
  }
  return null;
};

export const Component: React.FC = () => {
  return (
    <>
      <Background />
      <main className={styles.main}>
        <div className={styles.container}>
          <Space size={16}>
            <CustomSelect
              title="服务器"
              showSearch
              defaultValue="lucy"
              style={{ width: 200 }}
              options={[
                { value: "jack", label: "Jack" },
                { value: "lucy", label: "Lucy" },
                { value: "Yiminghe", label: "yiminghe" },
                { value: "disabled", label: "Disabled", disabled: true },
              ]}
            />
            <CustomSelect
              title="卡组"
              showSearch
              defaultValue="lucy"
              style={{ width: 200 }}
              options={[
                { value: "jack", label: "Jack" },
                { value: "lucy", label: "Lucy" },
                { value: "Yiminghe", label: "yiminghe" },
                { value: "disabled", label: "Disabled", disabled: true },
              ]}
            />
          </Space>
          <div className={styles["mode-select"]}>
            <Mode
              title="竞技匹配"
              desc="与世界上其他玩家在线匹配，您的排名将实时显示在排行榜上。"
              icon={<IconFont type="icon-battle" size={32} />}
            />
            <Mode
              title="休闲匹配"
              desc="使用任意卡组进行对战，将胜负暂且搁置，尽情享受决斗的乐趣。"
              icon={<IconFont type="icon-coffee" size={28} />}
            />
            <Mode
              title="单人模式"
              desc="开启与AI的决斗，验证自己的卡组，或者只是打发时间。"
              icon={<IconFont type="icon-chip" size={26} />}
            />
            <Mode
              title="自定义房间"
              desc="创建一个自定义的对战房间，便捷地与好友进行对战，甚至是举办一场竞技比赛。"
              icon={<SettingFilled />}
            />
            <Mode
              title="录像回放"
              desc="自由查看进行过的决斗，回味那些精彩的逆转瞬间。"
              icon={<IconFont type="icon-record" size={24} />}
            />
            <Mode
              title="卡组编辑"
              desc="创建和编辑卡组，在上万种卡片中选择，组建独一无二的构筑。"
              icon={<EditFilled />}
            />
          </div>
        </div>
      </main>
    </>
  );
};
Component.displayName = "Match";

const Mode: React.FC<{
  title: string;
  desc: string;
  icon: React.ReactNode;
  onClick?: () => void;
}> = ({ title, desc, icon, onClick }) => (
  <div className={styles.mode} onClick={onClick}>
    <div className={styles.icon}>{icon}</div>
    <div className={styles.title}>{title}</div>
    <div className={styles.desc}>{desc}</div>
  </div>
);

/** 从SSO跳转回的URL之中，解析用户信息 */
function getSSOUser(searchParams: URLSearchParams): User {
  return Object.fromEntries(searchParams) as unknown as User;
}

const CustomSelect: React.FC<
  React.ComponentProps<typeof Select> & { title: string }
> = ({ title, className, dropdownStyle, ...rest }) => (
  <ConfigProvider
    theme={{
      components: {
        Select: {
          colorBgElevated: "hsla(0, 0%, 100%, 0.05)",
          controlItemBgActive: "#79797955",
          colorBorder: "transparent",
          colorBgContainer: "hsla(0, 0%, 100%, 0.05)",
          colorPrimaryHover: "#3400d1",
          lineWidth: 0,
        },
      },
    }}
  >
    <div className={styles["custom-select"]}>
      <span className={styles.prefix}>{title}</span>
      <Select
        className={classNames(styles.select, className)}
        size="large"
        dropdownStyle={{
          backdropFilter: "blur(20px)",
          ...dropdownStyle,
        }}
        {...rest}
      />
    </div>
  </ConfigProvider>
);
