/* eslint-disable no-dupe-class-members */
import Client, { CloseReason } from "@network/Client";
import Bridge from "@network/Bridge";

import Room from "@game/Room";

export interface Connection {
    client: Client;
    server: Bridge;
    room: Room;
    deck: Buffer;
    timeout?: NodeJS.Timeout;
}

export default class ConnectionManager {
    private static readonly connections: Map<string, Connection> = new Map<string, Connection>();

    public static get(client: Client) {
        return this.connections.get(client.id);
    }
    public static has(client: Client) {
        return this.connections.has(client.id);
    }
    public static set(client: Client, connection: Connection) {
        this.connections.set(client.id, connection);
    }
    public static remove(client: Client) {
        this.connections.delete(client.id);
    }

    public static release(connection: Client, reconnected?: boolean): void;
    public static release(connection: Connection, reconnected?: boolean): void;
    public static release(target: Client | Connection, reconnected?: boolean) {
        if (target instanceof Client) {
            const connection = ConnectionManager.get(target);
            if (!connection) {
                return;
            }

            // eslint-disable-next-line no-param-reassign
            target = connection;
        }

        if (target.client && target.client.closed && !reconnected) {
            target.client.close(CloseReason.Unknown);
        }
        if (target.server && target.server.closed && !reconnected) {
            target.server.close(true);
        }

        if (target.timeout) {
            clearTimeout(target.timeout);
        }
    }

    public static clearConnectionsByRoom(room: Room) {
        ConnectionManager.connections.forEach(connection => {
            if (connection.room !== room) {
                return;
            }

            ConnectionManager.release(connection);
            ConnectionManager.remove(connection.client);
        });
    }
}
