/* eslint-disable no-bitwise */
import Client, { CloseReason } from "@network/Client";
import Packet from "@network/Packet";
import PacketBuilder from "@network/PacketBuilder";
import { PacketType } from "@network/message";
import { ServerToClientMessageType, TimeLimitMessageData } from "@network/message/stoc";
import { ClientToServerMessageType, TimeConfirmMessageData } from "@network/message/ctos";

import Room from "@game/Room";

import { GameType, PacketData } from "@root/types";
import { GameMessageType, RoomState } from "@root/constants";

const LONG_RESOLVE_CARDS = [11110587, 32362575, 43040603, 58577036, 79106360];

export default class Heartbeat {
    private static readonly instances: Map<Client, Heartbeat> = new Map<Client, Heartbeat>();
    private static remove(client: Client) {
        Heartbeat.instances.delete(client);
    }

    public static get(client: Client) {
        let heartbeat = Heartbeat.instances.get(client);
        if (!heartbeat) {
            heartbeat = new Heartbeat(client);
            Heartbeat.instances.set(client, heartbeat);
        }

        return heartbeat;
    }

    private readonly client: Client;

    private heartbeatActivating: boolean;
    private heartbeatTimeout: NodeJS.Timeout | null;
    private heartbeatAnswered: boolean;
    private isFirst: boolean;

    public constructor(client: Client) {
        this.client = client;

        this.heartbeatActivating = false;
        this.heartbeatAnswered = false;
        this.heartbeatTimeout = null;

        this.client.on("close", this.onClose.bind(this));
        this.client.on("send", this.onSend.bind(this));
        this.client.on("receive", this.onReceive.bind(this));
    }

    private onClose() {
        this.release();
    }
    private onReceive(packet: Packet<PacketType.ServerToClient>, packetData: PacketData<PacketType.ServerToClient>) {
        switch (packetData.type) {
            case ServerToClientMessageType.GAME_MSG:
                this.onGameMessage(packet.buffer.readInt8(0), packet.buffer);
                break;

            case ServerToClientMessageType.TIME_LIMIT:
                this.onTimeLimit(packetData);
                break;

            default:
                break;
        }
    }
    private onSend(_: Packet<PacketType.ClientToServer>, packetData: PacketData<PacketType.ClientToServer>) {
        switch (packetData.type) {
            case ClientToServerMessageType.TIME_CONFIRM:
                this.onTimeConfirm(packetData);
                break;

            default:
                break;
        }
    }

    private onGameMessage = (type: GameMessageType, buffer: Buffer) => {
        const { currentRoom } = this.client;
        if (!currentRoom) {
            return;
        }

        switch (type) {
            case GameMessageType.START:
                this.isFirst = !(buffer.readUInt8(1) & 0xf);
                break;

            case GameMessageType.WIN:
                currentRoom.connectedClients.forEach(client => {
                    // eslint-disable-next-line no-param-reassign
                    Heartbeat.get(client).heartbeatActivating = false;
                });

                currentRoom.longResolveChain = null;
                currentRoom.longResolveCard = null;
                break;

            case GameMessageType.CONFIRM_CARDS:
                {
                    const count = buffer.readInt8(2);
                    let check = false;
                    let foundDeckCount = 0;
                    let foundLimboCount = 0;

                    // eslint-disable-next-line no-multi-assign
                    for (let i = 3, n = 3, ref5 = 3 + (count - 1) * 7; n <= ref5; i = n += 7) {
                        const location = buffer.readInt8(i + 5);
                        if ((location & 0x41) > 0) {
                            foundDeckCount++;
                        } else if (location === 0) {
                            foundLimboCount++;
                        }

                        if ((foundDeckCount > 0 && count > 1) || foundLimboCount > 0) {
                            check = true;
                            break;
                        }
                    }

                    if (check) {
                        this.heartbeatActivating = true;
                    }
                }
                break;

            default:
                break;
        }

        if (!this.client.closed && this.client.position === 0) {
            if (type === GameMessageType.CHAINING) {
                const card = buffer.readUInt32LE(1);
                const found = LONG_RESOLVE_CARDS.some(id => id === card);

                if (found) {
                    currentRoom.longResolveCard = card;
                } else {
                    currentRoom.longResolveCard = null;
                }
            } else if (type === GameMessageType.CHAINED) {
                const chain = buffer.readInt8(1);
                if (!currentRoom.longResolveChain) {
                    currentRoom.longResolveChain = [];
                }

                currentRoom.longResolveChain[chain] = true;
                delete currentRoom.longResolveCard;
            } else if (type === GameMessageType.CHAIN_SOLVING && currentRoom.longResolveChain) {
                const chain = buffer.readInt8(1);
                if (currentRoom.longResolveChain[chain]) {
                    currentRoom.connectedClients.forEach(client => {
                        Heartbeat.get(client).heartbeatActivating = true;
                    });
                }
            } else if (
                (type === GameMessageType.CHAIN_NEGATED || type === GameMessageType.CHAIN_DISABLED) &&
                currentRoom.longResolveChain
            ) {
                const chain = buffer.readInt8(1);
                delete currentRoom.longResolveChain[chain];
            } else if (type === GameMessageType.CHAIN_END) {
                currentRoom.longResolveCard = null;
                currentRoom.longResolveChain = null;
            }
        }
    };
    private onTimeLimit = ({ player }: TimeLimitMessageData) => {
        const room = Room.findConnectedRoom(this.client);
        if (!room) {
            return;
        }

        if (!(room.status === RoomState.Dueling && !room.hasBot)) {
            return;
        }

        let check: boolean;
        if (room.gameOptions.type !== GameType.Tag) {
            check = (this.isFirst && player === 0) || (!this.isFirst && player === 1);
        } else {
            const cur_players = [];
            switch (room.turn % 4) {
                case 1:
                    cur_players[0] = 0;
                    cur_players[1] = 3;
                    break;

                case 2:
                    cur_players[0] = 0;
                    cur_players[1] = 2;
                    break;

                case 3:
                    cur_players[0] = 1;
                    cur_players[1] = 2;
                    break;

                case 0:
                    cur_players[0] = 1;
                    cur_players[1] = 3;
                    break;

                default:
                    break;
            }

            const firstClient = room.connectedClients[0];
            if (!Heartbeat.get(firstClient).isFirst) {
                cur_players[0] += 2;
                cur_players[1] -= 2;
            }

            check = this.client.position === cur_players[player];
        }

        if (check) {
            this.register(false);
        }
    };
    private onTimeConfirm = (_data: TimeConfirmMessageData) => {
        this.heartbeatActivating = false;
        this.heartbeatAnswered = true;
        this.unregister();
    };

    private unregister = () => {
        if (!this.heartbeatTimeout) {
            return false;
        }

        clearTimeout(this.heartbeatTimeout);
        this.heartbeatTimeout = null;
        return true;
    };
    public register = (send: boolean) => {
        const room = Room.findConnectedRoom(this.client);
        if (room) {
            return false;
        }

        if (this.client.closed || this.client.position > 3 || this.heartbeatActivating) {
            return false;
        }

        if (this.heartbeatTimeout) {
            this.unregister();
        }

        this.heartbeatAnswered = false;

        if (send) {
            this.client.send(
                PacketBuilder.build(PacketType.ServerToClient)
                    .type(ServerToClientMessageType.TIME_LIMIT)
                    .add("unsigned char", 0)
                    .add("unsigned short", 0)
                    .build(),
            );

            this.client.send(
                PacketBuilder.build(PacketType.ServerToClient)
                    .type(ServerToClientMessageType.TIME_LIMIT)
                    .add("unsigned char", 1)
                    .add("unsigned short", 0)
                    .build(),
            );
        }

        this.heartbeatTimeout = setTimeout(() => {
            this.unregister();
            if (!(this.client.closed || this.heartbeatAnswered)) {
                this.client.close(CloseReason.Heartbeat);
            }
        }, 10000);

        return true;
    };

    private release() {
        this.unregister();

        this.client.removeListener("close", this.onClose);
        this.client.removeListener("send", this.onSend);
        this.client.removeListener("receive", this.onReceive);

        Heartbeat.remove(this.client);
    }
}
