/* eslint-disable no-dupe-class-members */
import Packet from "@network/Packet";
import { PacketType } from "@network/message";
import { PacketDataType } from "@network/message/structure";
import { ClientToServerMessageType } from "@network/message/ctos";
import { ServerToClientMessageType } from "@network/message/stoc";

export default class PacketBuilder<Type extends PacketType.ServerToClient | PacketType.ClientToServer> {
    public static build<Type extends PacketType.ServerToClient | PacketType.ClientToServer>(
        type: Type,
    ): PacketBuilder<Type> {
        return new PacketBuilder(type);
    }

    private readonly data: Buffer[];
    private messageType: ClientToServerMessageType | ServerToClientMessageType;
    private packetType: Type;

    private constructor(type: Type) {
        this.data = [];
        this.packetType = type;
    }

    public type(type: ClientToServerMessageType | ServerToClientMessageType): PacketBuilder<Type> {
        this.messageType = type;
        return this;
    }

    public add(type: "unsigned int", value: number, length?: 1): PacketBuilder<Type>;
    public add(type: "unsigned int", value: number[], length: number): PacketBuilder<Type>;
    public add(type: "unsigned short", value: number, length?: 1): PacketBuilder<Type>;
    public add(type: "unsigned short", value: number[], length: number): PacketBuilder<Type>;
    public add(type: "unsigned char", value: number, length?: 1): PacketBuilder<Type>;
    public add(type: "unsigned char", value: number[], length: number): PacketBuilder<Type>;
    public add(type: "string", value: string, length?: number): PacketBuilder<Type>;

    public add(type: PacketDataType, value: number | number[] | string, length?: 1 | number): PacketBuilder<Type> {
        if (type !== "string" && typeof value !== "string") {
            let byteCount = 0;
            switch (type) {
                case "unsigned char":
                    byteCount = 1;
                    break;

                case "unsigned short":
                    byteCount = 2;
                    break;

                case "unsigned int":
                    byteCount = 4;
                    break;

                default:
                    throw new Error(`Unknown packet data type: ${type}`);
            }

            let bufferSize = byteCount;
            if (length) {
                bufferSize *= length;
            }

            const buffer = Buffer.alloc(bufferSize);
            if (Array.isArray(value)) {
                value.forEach((v, i) => {
                    buffer.writeIntLE(v, i * byteCount, byteCount);
                });
            } else {
                buffer.writeIntLE(value, 0, byteCount);
            }

            this.data.push(buffer);
        } else if (type === "string" && typeof value === "string") {
            let targetValue = `${value}\0`;
            if (length && targetValue.length > length) {
                targetValue = targetValue.substring(0, length);
            }

            const buffer = Buffer.alloc((length || targetValue.length) * 2, 0);
            buffer.write(targetValue, "utf16le");

            this.data.push(buffer);
        } else {
            throw new Error("Invalid arguments");
        }

        return this;
    }

    public buffer(data: Buffer) {
        this.data.push(data);
        return this;
    }

    public build(): Packet<Type> {
        const data = Buffer.concat(this.data);
        const packetHeader = Buffer.alloc(3, 0);
        packetHeader.writeUInt16LE(data.length + 1, 0);
        packetHeader.writeUInt8(this.messageType as number, 2);

        return Packet.parse<Type>(Buffer.concat([packetHeader, data]), this.packetType)[0];
    }
}
