import { BaseMessageData, PacketType } from "@network/message";
import getMessageStructure, { MessageStructureContainer } from "@network/message/structure";
import { ClientToServerMessages } from "@network/message/ctos";
import { ServerToClientMessages } from "@network/message/stoc";

export default class Packet<PacketType extends PacketType.ServerToClient | PacketType.ClientToServer> {
    public static parse<T extends PacketType.ServerToClient | PacketType.ClientToServer>(
        data: Buffer,
        packetType: T,
    ): Packet<T>[] {
        const result: Packet<T>[] = [];

        // eslint-disable-next-line no-constant-condition
        while (true) {
            if (data.length < 2 || data.length < 3) {
                break;
            }

            const messageLength = data.readUInt16LE(0);
            const messageType = data.readUInt8(2) as keyof MessageStructureContainer[T];
            if (data.length < 2 + messageLength) {
                break;
            }

            const packet = new Packet<T>(messageType, messageLength, data.slice(0, 2 + messageLength), packetType);
            result.push(packet);

            // eslint-disable-next-line no-param-reassign
            data = data.slice(2 + messageLength);
        }

        return result;
    }

    public readonly type: keyof MessageStructureContainer[PacketType];
    public readonly length: number;
    public readonly packetType: PacketType;
    public readonly buffer: Buffer;
    public readonly dataBuffer: Buffer;
    private cursorOffset: number;

    private constructor(
        messageType: keyof MessageStructureContainer[PacketType],
        messageLength: number,
        buffer: Buffer,
        packetType: PacketType,
    ) {
        this.type = messageType;
        this.length = messageLength;
        this.buffer = buffer;
        this.dataBuffer = buffer.slice(3, messageLength - 1 + 3);
        this.packetType = packetType;
        this.cursorOffset = 0;
    }

    private retrieveStringField = (length: number, offset?: number) => {
        const targetOffset = offset || this.cursorOffset;
        if (!offset) {
            this.cursorOffset += length;
        }

        const result = this.dataBuffer.toString("UTF-16LE", targetOffset, targetOffset + length);
        const stringLength = result.indexOf("\0");

        if (stringLength === -1) {
            return result;
        }

        return result.slice(0, stringLength);
    };
    private retrieveNumericField = (byteCount: number, length: number = 1, offset?: number) => {
        const targetOffset = offset || this.cursorOffset;
        if (!offset) {
            this.cursorOffset += byteCount * length;
        }

        if (length === 1) {
            return this.dataBuffer.readUIntLE(targetOffset, byteCount);
        }

        const result = [];
        for (let i = 0; i < length; ++i) {
            result.push(this.dataBuffer.readUIntLE(targetOffset + i * byteCount, byteCount));
        }

        return result;
    };

    public process = <PT extends PacketType.ServerToClient | PacketType.ClientToServer = PacketType>() => {
        const messageStructure = getMessageStructure(this.packetType, this.type);
        if (!messageStructure) {
            const base: BaseMessageData<string> = {
                type: this.type as any,
                bypass: true,
            };

            return base;
        }

        const result: BaseMessageData<any> = {
            type: this.type as any,
            bypass: false,
        };

        messageStructure.forEach(item => {
            let value: any;
            switch (item.type) {
                case "unsigned int":
                    value = this.retrieveNumericField(4, item.length);
                    break;

                case "unsigned short":
                    value = this.retrieveNumericField(2, item.length);
                    break;

                case "unsigned char":
                    value = this.retrieveNumericField(1, item.length);
                    break;

                case "string":
                    if (item.length) {
                        value = this.retrieveStringField(item.length * 2, item.offset);
                    }
                    break;

                default:
                    break;
            }

            (result as any)[item.name] = value;
        });

        return (result as any) as PT extends PacketType.ClientToServer
            ? ClientToServerMessages
            : ServerToClientMessages;
    };
}
