/* eslint-disable no-bitwise,no-param-reassign,no-dupe-class-members */
import net from "net";
import { EventEmitter } from "tsee";

import Room from "@game/Room";
import Bridge from "@network/Bridge";
import Packet from "@network/Packet";
import PacketBuilder from "@network/PacketBuilder";
import { PacketType } from "@network/message";
import {
    ChatMessageData,
    ClientToServerMessageType,
    JoinGameMessageData,
    KickMessageData,
    PlayerInfoMessageData,
} from "@network/message/ctos";
import { ServerToClientMessageType, TypeChangeMessageData } from "@network/message/stoc";

import Reconnect from "@plugins/reconnect";

import logger from "@utils/logger";

import { ChatType, GameMessageType, RoomState } from "@root/constants";
import { PacketData } from "@root/types";
import ClientObject from "@api/Client";

export enum CloseReason {
    Unknown,
    Heartbeat,
    Errored,
    Timeout,
    Kicked,
    ReconnectTimeout,
    Destroyed,
    TimeoutKicked,
}

export type ClientEvents = {
    close: () => void;
    send: (packet: Packet<PacketType.ClientToServer>, packetData: PacketData<PacketType.ClientToServer>) => void;
    receive: (packet: Packet<PacketType.ServerToClient>, packetData: PacketData<PacketType.ServerToClient>) => void;
};

//
// This is a class for client of **THIS** server.
export default class Client {
    private static connectedClients: Client[] = [];

    public static dumpAllClients() {
        return Client.connectedClients.map(client => {
            return client.toData();
        });
    }

    public static broadcastMessage(message: string) {
        Client.connectedClients.forEach(client => {
            client.sendChat(`[Server]: ${message}`, ChatType.PINK);
        });

        return Client.connectedClients.length;
    }

    private ygoproServer: Bridge;

    private isFirst: boolean;
    private socket: net.Socket;
    private closeReason: CloseReason;
    private emitter: EventEmitter<ClientEvents>;
    private preduel: boolean;

    private _name: string;
    private _closed: boolean;
    private _position: number;
    private _host: boolean;
    private _deckBuffer: Buffer;

    public constructor(socket: net.Socket) {
        this.socket = socket;
        this.closeReason = CloseReason.Unknown;
        this._closed = false;
        this.isFirst = false;
        this.emitter = new EventEmitter<ClientEvents>();
        this._position = -1;
        this._host = false;

        this.ygoproServer = new Bridge();
        this.ygoproServer.receive(this.onReceive);
        this.ygoproServer.addPipeline(this.onRawReceive);

        Reconnect.install(this, this.ygoproServer);

        Client.connectedClients.push(this);

        this.socket.setKeepAlive(true, 2000);
        this.socket.on("error", this.onError);
        this.socket.on("close", this.onClose);
        this.socket.on("data", this.onData);
    }

    public get name(): string {
        return this._name;
    }

    public get position(): number {
        return this._position;
    }

    public get isHost(): boolean {
        return this._host;
    }

    public get currentRoom(): Room | undefined {
        return Room.findConnectedRoom(this);
    }

    public get closed(): boolean {
        return this._closed;
    }

    public get id(): string {
        return `${this.socket.remoteAddress}:${this._name}`;
    }

    public get selectedPreduel(): boolean {
        return this.preduel;
    }

    public get deckBuffer(): Buffer {
        return this._deckBuffer;
    }

    public get isAI() {
        return this.socket.remoteAddress ? this.socket.remoteAddress.indexOf("127.0.0.1") >= 0 : false;
    }

    // controls
    private kick = () => {
        const reconnect = Reconnect.get(this);
        if (reconnect && this.closed) {
            if (this.ygoproServer.closed && !reconnect.hadNewConnection) {
                this.ygoproServer.close(true);
            }
        } else {
            this.socket.destroy();
        }
    };
    private changePreduelState = (state: boolean) => {
        const room = this.currentRoom;
        if (!room) {
            return;
        }

        this.preduel = state;
    };
    public replaceClient = (client: Client) => {
        if (!this.currentRoom) {
            return;
        }

        this.currentRoom.replacePlayer(client, this);

        this._name = client._name;
        this._position = client._position;
        this._host = client._host;
        this._closed = client._closed;

        this.isFirst = client.isFirst;
        this.preduel = client.preduel;
        this.closeReason = client.closeReason;
    };
    public replaceServer = (server: Bridge) => {
        const oldServer = this.ygoproServer;
        oldServer.removeAllPipelines();

        this.ygoproServer = server;
        this.ygoproServer.removeAllPipelines();
        this.ygoproServer.addPipeline(this.onRawReceive);

        return oldServer;
    };
    public on = <Key extends keyof ClientEvents>(name: Key, listener: ClientEvents[Key]) => {
        this.emitter.on(name, listener);
    };
    public once = <Key extends keyof ClientEvents>(name: Key, listener: ClientEvents[Key]) => {
        this.emitter.once(name, listener);
    };
    public removeListener = <Key extends keyof ClientEvents>(name: Key, listener: ClientEvents[Key]) => {
        this.emitter.removeListener(name, listener);
    };

    // networking
    private write = (buffer: Buffer) => {
        // refuse if socket is already closed.
        if (!this.socket.writable) {
            return;
        }

        this.socket.write(buffer);
    };
    public close = (reason: CloseReason, force: boolean = false) => {
        // eslint-disable-next-line default-case
        switch (reason) {
            case CloseReason.TimeoutKicked:
                this.sendChat(`[Server]: 게임을 시작하지 않아 강제 퇴장 당하셨습니다.`, ChatType.RED);
                break;
        }

        return new Promise(resolve => {
            this.closeReason = reason;
            if (!force) {
                this.socket.end(() => {
                    resolve();
                });
            } else {
                this.socket.destroy();
                resolve();
            }
        });
    };

    public sendChat = (message: string, type: ChatType = ChatType.LIGHTBLUE) => {
        const packet = PacketBuilder.build(PacketType.ServerToClient)
            .type(ServerToClientMessageType.CHAT)
            .add("unsigned short", type)
            .add("string", message, 255)
            .build();

        this.write(packet.buffer);
    };

    public send(buffer: Buffer): void;
    public send(packet: Packet<PacketType.ServerToClient>): void;
    public send(data: Buffer | Packet<PacketType.ServerToClient>) {
        if (data instanceof Packet) {
            this.socket.write(data.buffer);
            return;
        }

        this.socket.write(data);
    }

    // socket event handlers
    private onError = (_error: Error) => {
        this.closeReason = CloseReason.Errored;
        this.onClose(true);
    };
    private onClose = (_hadError: boolean) => {
        if (!this._closed) {
            Client.connectedClients = Client.connectedClients.filter(c => c !== this);

            this._closed = true;
            this.emitter.emit("close");

            const room = this.currentRoom;
            const reconnect = Reconnect.get(this);
            if (room && reconnect && this.closeReason !== CloseReason.Destroyed) {
                if (!reconnect.register()) {
                    room.disconnect(this, this.closeReason);
                    this.ygoproServer.close(true);
                }
            } else if (!reconnect || !reconnect.hadNewConnection) {
                this.ygoproServer.close(true);
            }
        }
    };
    private onData = (data: Buffer) => {
        const packets = Packet.parse(data, PacketType.ClientToServer);
        if (__DEV__) {
            const debugInfo = packets
                .map(packet => {
                    return `[type: ${packet.type} (${ClientToServerMessageType[packet.type]})]`;
                })
                .join(", ");

            logger.debug(`Received ${packets.length} client to server packets: ${debugInfo}`);
        }

        const skippedPackets: Array<boolean | undefined> = packets.map(() => false);
        packets.forEach((packet, index) => {
            skippedPackets[index] = this.onSend(packet);
        });
        packets.forEach((packet, index) => {
            if (skippedPackets[index]) {
                return;
            }

            this.ygoproServer.write(packet.buffer);
        });
    };

    // high-level message transfer event handlers
    private onSend = (packet: Packet<PacketType.ClientToServer>): boolean | undefined => {
        const packetData = packet.process();
        switch (packetData.type) {
            case ClientToServerMessageType.CHAT:
                if (!this.onSendChat(packetData)) {
                    return true;
                }

                break;

            case ClientToServerMessageType.PLAYER_INFO:
                this.onPlayerInformation(packetData);
                break;

            case ClientToServerMessageType.HS_KICK:
                this.onKickClient(packetData);
                break;

            case ClientToServerMessageType.JOIN_GAME:
                this.onJoinGameRequested(packetData);
                break;

            case ClientToServerMessageType.TIME_CONFIRM:
                break;

            case ClientToServerMessageType.UPDATE_DECK:
                this.onUpdateDeck(packet.dataBuffer);
                break;

            case ClientToServerMessageType.HAND_RESULT:
                this.changePreduelState(true);
                break;

            case ClientToServerMessageType.TP_RESULT:
                this.changePreduelState(true);
                break;

            default:
                if (packetData.bypass) {
                    return undefined;
                }

                throw new Error(
                    `Unhandled CTOS message type: ${packetData.type} (${ClientToServerMessageType[packet.type]})`,
                );
        }

        this.emitter.emit("send", packet, packetData);
        return undefined;
    };
    private onReceive = (packet: Packet<PacketType.ServerToClient>) => {
        const packetData = packet.process();

        switch (packetData.type) {
            case ServerToClientMessageType.TIME_LIMIT:
            case ServerToClientMessageType.DUEL_START:
            case ServerToClientMessageType.JOIN_GAME:
            case ServerToClientMessageType.HS_PLAYER_CHANGE:
                break;

            case ServerToClientMessageType.SELECT_TP:
                this.changePreduelState(false);
                break;

            case ServerToClientMessageType.GAME_MSG:
                this.onGameMessage(packet.buffer.readInt8(0), packet.buffer);
                break;

            case ServerToClientMessageType.TYPE_CHANGE:
                this.onPlayerTypeChange(packetData);
                break;

            case ServerToClientMessageType.CHANGE_SIDE:
                this.onChangeSide();
                break;

            case ServerToClientMessageType.SELECT_HAND:
                this.changePreduelState(false);
                break;

            default:
                if (packetData.bypass) {
                    return;
                }

                throw new Error(
                    `Unhandled STOC message type: ${packetData.type} (${ClientToServerMessageType[packet.type]})`,
                );
        }

        this.emitter.emit("receive", packet, packetData);
    };
    private onRawReceive = (packet: Packet<PacketType.ServerToClient>) => {
        this.write(packet.buffer);
    };

    // CTOS packet message handlers
    private onJoinGameRequested = async ({ version, password: code }: JoinGameMessageData) => {
        if (version !== 0x1351) {
            this.sendChat("[Server]: 클라이언트 버전이 맞지 않습니다. 클라이언트를 업데이트 해주세요.", ChatType.RED);
            this.send(
                PacketBuilder.build(PacketType.ServerToClient)
                    .type(ServerToClientMessageType.ERROR_MSG)
                    .add("unsigned char", 4)
                    .add("unsigned char", 0)
                    .add("unsigned char", 0)
                    .add("unsigned char", 0)
                    .add("unsigned int", 0x1351)
                    .build(),
            );

            this.close(CloseReason.Unknown);
            return;
        }

        const reconnect = Reconnect.get(this);
        if (reconnect && reconnect.checkReconnectable()) {
            this.socket.setTimeout(300000);
            reconnect.reconnectToRoom();
            return;
        }

        const room = Room.findOrCreateRoom(code);
        if (!room.isStarted) {
            await room.start();
        }

        if (room.status === RoomState.Ready) {
            await this.ygoproServer.open(room);
        }

        room.connect(this);
    };
    private onKickClient = async ({ position }: KickMessageData) => {
        const room = Room.findConnectedRoom(this);
        if (!room) {
            return;
        }

        const client = room.connectedClients.find(({ position: pos }) => pos === position);
        if (!client) {
            return;
        }

        client.sendChat("[Server]: 대기실에서 강제 퇴장 당하셨습니다.", ChatType.RED);
        await client.close(CloseReason.Kicked);
    };
    private onPlayerInformation = ({ name }: PlayerInfoMessageData) => {
        this._name = name;
    };
    private onUpdateDeck = (deckBuffer: Buffer) => {
        const reconnect = Reconnect.get(this);
        if (reconnect && reconnect.isReconnectedToRoom) {
            if (!reconnect.checkReconnectable()) {
                this.sendChat("[Server]: 이전 게임에 재접속 하는데 실패하였습니다.", ChatType.RED);
                this.kick();
            } else if (reconnect.checkReconnectable(deckBuffer)) {
                reconnect.reconnectToGame();
            } else {
                this.sendChat(
                    "[Server]: 이전 게임에서 사용했던 덱과 일치하지 않습니다. 같은 덱을 선택 해주세요.",
                    ChatType.RED,
                );

                this.send(
                    PacketBuilder.build(PacketType.ServerToClient)
                        .type(ServerToClientMessageType.ERROR_MSG)
                        .add("unsigned char", 2)
                        .add("unsigned char", 0)
                        .add("unsigned char", 0)
                        .add("unsigned char", 0)
                        .add("unsigned int", 0)
                        .build(),
                );

                this.send(
                    PacketBuilder.build(PacketType.ServerToClient)
                        .type(ServerToClientMessageType.HS_PLAYER_CHANGE)
                        .add("unsigned char", (this.position << 4) | 0xa)
                        .build(),
                );
            }

            return;
        }

        const room = this.currentRoom;
        if (!room) {
            return;
        }

        if (reconnect) {
            this._deckBuffer = deckBuffer;
        }

        if (room.status !== RoomState.Ready) {
            this.changePreduelState(true);
        }
    };
    private onSendChat = ({ message }: ChatMessageData) => {
        const currentRoom = Room.findConnectedRoom(this);
        if (!currentRoom || !currentRoom.checkClientIsSpectator(this)) {
            return true;
        }

        currentRoom.broadcastChat(`[관전자] ${this._name}: ${message}`, ChatType.YELLOW, true);
        return false;
    };

    // STOC packet message handlers
    private onGameMessage = async (type: GameMessageType, buffer: Buffer) => {
        const reconnect = Reconnect.get(this);
        if (reconnect) {
            if (reconnect.checkReconnectable()) {
                const data = reconnect.reconnectToRoom();
                if (!data) {
                    this.sendChat(
                        "[Server]: 재접속을 시도하는 도중 알수 없는 오류가 발생했습니다. 관리자에게 문의하세요.",
                        ChatType.RED,
                    );

                    await this.close(CloseReason.Errored);
                    return;
                }

                this._position = data.newPosition;
            }
        }

        const currentRoom = Room.findConnectedRoom(this);
        if (!currentRoom) {
            return;
        }

        currentRoom.processGameMessage(type, buffer, this);
    };
    private onPlayerTypeChange = ({ value }: TypeChangeMessageData) => {
        this._position = value & 0xf;
        this._host = ((value >> 4) & 0xf) !== 0;
    };
    private onChangeSide = () => {
        const room = this.currentRoom;
        if (!room) {
            return;
        }

        room.changeSide(this);
        this.preduel = true;
    };

    public toData(): ClientObject {
        return {
            id: this.id,
            name: this.name,
            remoteAddress: this.socket.remoteAddress || "::1",
            isHost: this.isHost,
            position: this.position,
        };
    }
}
