/* eslint-disable import/prefer-default-export,no-cond-assign */
import * as _ from "lodash";
import moment from "moment";
import fs from "fs";

import { GameOption, GameType } from "@root/types";
import path from "path";

export interface WindbotInformation {
    name: string;
    deck: string;
    dialog: string;
}

interface LfListItem {
    date: moment.Moment;
    tcg: boolean;
}

const LFLISTS = (() => {
    const result: LfListItem[] = [];
    if (!fs.existsSync("ygopro/lflist.conf")) {
        return result;
    }

    const matches = fs.readFileSync("ygopro/lflist.conf", "utf8").match(/!.*/g);
    if (!matches) {
        throw new Error("Failed to parse lflist.conf!");
    }

    matches.forEach(list => {
        const date = list.match(/!([\d.]+)/);
        if (!date) {
            return;
        }

        const dateString = list.match(/!([\d.]+)/);
        if (!dateString) {
            throw new Error(`Failed to parse lflist.conf with date string: ${list}`);
        }

        result.push({
            date: moment(dateString[1], "YYYY.MM.DD").utcOffset("-08:00"),
            tcg: list.indexOf("TCG") !== -1,
        });
    });

    return result;
})();

export function parseRoomName(name: string) {
    const gameOptions: Partial<GameOption> = {};
    if (name === "AI") {
        gameOptions.rule = 2;
        gameOptions.lfList = -1;
        gameOptions.timeLimit = 999;
        gameOptions.isAIRoom = true;
        return gameOptions;
    }

    if (LFLISTS.length) {
        if (gameOptions.rule === 1 && gameOptions.lfList === 0) {
            gameOptions.lfList = _.findIndex(LFLISTS, list => list.tcg);
        }
    } else {
        gameOptions.lfList = -1;
    }

    let param: RegExpMatchArray | null;
    if (name.slice(0, 2) === "M#") {
        gameOptions.type = GameType.Match;
    } else if (name.slice(0, 2) === "T#") {
        gameOptions.type = GameType.Tag;
        gameOptions.lifePoints = 16000;
    } else if (name.slice(0, 3) === "AI#") {
        gameOptions.rule = 2;
        gameOptions.lfList = -1;
        gameOptions.timeLimit = 999;
        gameOptions.isAIRoom = true;
    } else if ((param = name.match(/^(\d)(\d)([TF])([TF])([TF])(\d+),(\d+),(\d+)/i))) {
        gameOptions.rule = parseInt(param[1], 10);
        gameOptions.type = parseInt(param[2], 10).toString() as GameType;
        gameOptions.duelRule = param[3] === "T" ? 3 : 4;
        gameOptions.noDeckCheck = param[4] === "T";
        gameOptions.noDeckShuffle = param[5] === "T";
        gameOptions.lifePoints = parseInt(param[6], 10);
        gameOptions.startHandCount = parseInt(param[7], 10);
        gameOptions.drawCount = parseInt(param[8], 10);
    } else if ((param = name.match(/(.+)#/)) !== null) {
        const rule = param[1].toUpperCase();
        if (rule.match(/(^|，|,)(M|MATCH)(，|,|$)/)) {
            gameOptions.type = GameType.Match;
        }

        if (rule.match(/(^|，|,)(T|TAG)(，|,|$)/)) {
            gameOptions.type = GameType.Tag;
            gameOptions.lifePoints = 16000;
        }

        if (rule.match(/(^|，|,)(TCGONLY|TO)(，|,|$)/)) {
            gameOptions.rule = 1;
            gameOptions.lfList = _.findIndex(LFLISTS, list => list.tcg);
        }

        if (rule.match(/(^|，|,)(OCGONLY|OO)(，|,|$)/)) {
            gameOptions.rule = 0;
            gameOptions.lfList = 0;
        }

        if (rule.match(/(^|，|,)(OT|TCG)(，|,|$)/)) {
            gameOptions.rule = 2;
        }

        if ((param = rule.match(/(^|，|,)LP(\d+)(，|,|$)/))) {
            gameOptions.lifePoints = _.clamp(parseInt(param[2], 10), 1, 99999);
        }

        if ((param = rule.match(/(^|，|,)(TIME|TM|TI)(\d+)(，|,|$)/))) {
            let timeLimit = parseInt(param[3], 10);
            if (timeLimit >= 1 && timeLimit <= 60) {
                timeLimit *= 60;
            }

            if (timeLimit < 0) {
                timeLimit = 180;
            } else if (timeLimit >= 999) {
                timeLimit = 999;
            }

            gameOptions.timeLimit = _.clamp(timeLimit, 180, 999);
        }

        if ((param = rule.match(/(^|，|,)(START|ST)(\d+)(，|,|$)/))) {
            gameOptions.startHandCount = _.clamp(parseInt(param[3], 10), 1, 40);
        }

        if ((param = rule.match(/(^|，|,)(DRAW|DR)(\d+)(，|,|$)/))) {
            gameOptions.drawCount = _.clamp(parseInt(param[3], 10), 1, 35);
        }

        if ((param = rule.match(/(^|，|,)(LFLIST|LF)(\d+)(，|,|$)/))) {
            gameOptions.lfList = parseInt(param[3], 10) - 1;
        }

        if (rule.match(/(^|，|,)(NOLFLIST|NF)(，|,|$)/)) {
            gameOptions.lfList = -1;
        }

        if (rule.match(/(^|，|,)(NOUNIQUE|NU)(，|,|$)/)) {
            gameOptions.rule = 3;
        }

        if (rule.match(/(^|，|,)(NOCHECK|NC)(，|,|$)/)) {
            gameOptions.noDeckCheck = true;
        }

        if (rule.match(/(^|，|,)(NOSHUFFLE|NS)(，|,|$)/)) {
            gameOptions.noDeckShuffle = true;
        }

        if (rule.match(/(^|，|,)(IGPRIORITY|PR)(，|,|$)/)) {
            gameOptions.duelRule = 4;
        }

        if ((param = rule.match(/(^|，|,)(DUELRULE|MR)(\d+)(，|,|$)/))) {
            const duelRule = parseInt(param[3], 10);
            if (duelRule && duelRule > 0 && duelRule <= 5) {
                gameOptions.duelRule = duelRule;
            }
        }

        if ((param = rule.match(/(^|，|,)(DEATH|DH)(\d*)(，|,|$)/))) {
            const deathTime = parseInt(param[3], 10);
            if (deathTime && deathTime > 0) {
                gameOptions.autoDeath = deathTime;
            } else {
                gameOptions.autoDeath = 40;
            }
        }
    }

    return gameOptions;
}

export const WINDBOT_PATH = (() => {
    if (!process.env.WINDBOT_PATH) {
        return path.join(process.cwd(), "windbot");
    }

    if (path.isAbsolute(process.env.WINDBOT_PATH)) {
        return process.env.WINDBOT_PATH;
    }

    return path.join(process.cwd(), process.env.WINDBOT_PATH);
})();

export const YGOPRO_PATH = (() => {
    if (!process.env.YGOPRO_PATH) {
        return path.join(process.cwd(), "ygopro");
    }

    if (path.isAbsolute(process.env.YGOPRO_PATH)) {
        return process.env.YGOPRO_PATH;
    }

    return path.join(process.cwd(), process.env.YGOPRO_PATH);
})();

export const WINDBOT_DEFINITIONS: WindbotInformation[] = JSON.parse(
    fs.readFileSync(path.join(WINDBOT_PATH, "./bots.json")).toString(),
).windbots;
