/* eslint-disable no-param-reassign,no-cond-assign,no-bitwise */
import * as _ from "lodash";
import ChildProcess from "child_process";
import generateUUID from "uuid/v4";
import request from "request";
import QueryString from "querystring";

import Client, { CloseReason } from "@network/Client";
import Packet from "@network/Packet";
import { PacketType } from "@network/message";
import { PlayerChangeMessageData, ServerToClientMessageType } from "@network/message/stoc";

import Recorder from "@game/Recorder";
import { parseRoomName, WINDBOT_DEFINITIONS, WindbotInformation, YGOPRO_PATH } from "@game/utils";

import Heartbeat from "@plugins/heartbeat";
import ConnectionManager from "@plugins/reconnect/ConnectionManager";
import Reconnect from "@plugins/reconnect";

import logger from "@utils/logger";

import RoomObject from "@api/Room";

import { GameOption, GameType, ObjectType } from "@root/types";
import { ChatType, GameMessageType, RoomState } from "@root/constants";
import PacketBuilder from "@network/PacketBuilder";
import Uploader from "@plugins/uploader";

export enum RoomType {
    Random,
    Private,
}

export enum ReadyState {
    NotReady,
    Ready,
    Spectating,
}

//
// This 'Room' class internally contains an instance of YGOPro server process.
// and this class will work like a server.
//
export default class Room {
    private static openedRooms: ObjectType<Room> = {};

    private static createAIRoom(code: string, parsedGameOptions: Partial<GameOption>): Room {
        return new Room(code, RoomType.Private, { type: GameType.Single, ...parsedGameOptions }, true);
    }
    private static findOrCreateRandomRoom(gameType: GameType): Room {
        let availableRoomId: string | null = null;
        Object.keys(Room.openedRooms).some(roomId => {
            const room = Room.openedRooms[roomId];
            const clients = room.connectedClients.filter(c => room.checkClientIsPlayer(c));
            let maximumPlayerCount = 2;
            if (room._gameOptions.type === GameType.Tag) {
                maximumPlayerCount = 4;
            }

            if (
                clients.length < maximumPlayerCount &&
                room._type === RoomType.Random &&
                room._gameOptions.type === gameType
            ) {
                availableRoomId = roomId;
                return true;
            }

            return false;
        });

        if (!availableRoomId) {
            let roomId = generateUUID();
            while (Room.openedRooms[roomId]) {
                roomId = generateUUID();
            }

            return new Room(roomId, RoomType.Random, {
                type: gameType,
                lifePoints: gameType === GameType.Tag ? 16000 : 8000,
            });
        }

        return Room.openedRooms[availableRoomId];
    }
    public static findOrCreateRoom(name: string): Room {
        name = name.trim().toUpperCase();

        if (name === "" || name === "S" || name === "M" || name === "T") {
            let gameType: GameType;
            switch (name) {
                case "M":
                    gameType = GameType.Match;
                    break;

                case "T":
                    gameType = GameType.Tag;
                    break;

                default:
                    gameType = GameType.Single;
                    break;
            }

            return Room.findOrCreateRandomRoom(gameType);
        }

        if (Room.openedRooms[name]) {
            return Room.openedRooms[name];
        }

        const parsedGameOptions = parseRoomName(name);
        if (parsedGameOptions.isAIRoom) {
            return Room.createAIRoom(name, parsedGameOptions);
        }

        return new Room(name, RoomType.Private, parsedGameOptions);
    }
    public static findConnectedRoom(client: Client) {
        return Object.values(this.openedRooms).find(room => Boolean(room.clients.has(client)));
    }
    public static dumpAllRooms(): RoomObject[] {
        return Object.values(Room.openedRooms).map(room => room.toData());
    }
    public static getOpenedRoomCount(): number {
        return Object.keys(Room.openedRooms).length;
    }
    public static findRoomById(id: string): Room | null {
        const room = Room.openedRooms[id];
        return !room ? null : room;
    }

    public readonly host: string = "127.0.0.1";

    public longResolveCard: number | null;
    public longResolveChain: boolean[] | null;

    private readonly _gameOptions: GameOption;
    private readonly bot: WindbotInformation | undefined;
    private readonly id: string;
    private readonly heartbeatInterval: NodeJS.Timeout;
    private readonly _type: RoomType;
    private clients: Set<Client>;
    private serverProcess: ChildProcess.ChildProcessWithoutNullStreams | undefined;
    private readyState: ObjectType<ReadyState>;
    private serverProcessId: number;
    private portNumber: number;
    private recorder: Recorder;
    private _tpClient: Client;
    private _status: RoomState;
    private _turn: number;
    private _closed: boolean;
    private hostKickTimeout: NodeJS.Timeout | undefined;
    private readiedPlayerCountWithoutHost: number;

    private constructor(id: string, type: RoomType, gameOptions: Partial<GameOption> = {}, bot: boolean = false) {
        if (bot) {
            id = `AI#${_.random(0, 100000)
                .toString()
                .padStart(6, "0")}`;

            this.bot = _.sample(WINDBOT_DEFINITIONS);
        }

        Room.openedRooms[id] = this;
        this.clients = new Set<Client>();
        this.id = id;
        this._type = type;
        this.longResolveCard = null;
        this.longResolveChain = null;
        this._status = RoomState.Ready;
        this._turn = 0;
        this.readyState = {};
        this._gameOptions = _.defaults(gameOptions, {
            lfList: process.env.__LFLIST_INDEX__ || 0,
            rule: 0,
            type: GameType.Single,
            duelRule: process.env.__DUEL_RULE__ || "F",
            noDeckCheck: false,
            noDeckShuffle: false,
            lifePoints: 8000,
            startHandCount: 5,
            drawCount: 1,
            timeLimit: 300,
            autoDeath: 40,
            isAIRoom: false,
        });
        this.recorder = new Recorder();
        this.recorder.on("receive", this.onRecorderReceive);
        this.recorder.on("end", this.onRecorderEnd);

        this.heartbeatInterval = setInterval(() => {
            if (
                this.status !== RoomState.Ready &&
                (this._gameOptions.timeLimit === 0 || this.status !== RoomState.Dueling) &&
                !this.hasBot
            ) {
                this.clients.forEach(client => {
                    if (client && this.status !== RoomState.Siding) {
                        Heartbeat.get(client).register(true);
                    }
                });
            }
        }, 20000);
    }

    public get port(): number {
        return this.portNumber;
    }
    public get isStarted(): boolean {
        return Boolean(this.serverProcess);
    }
    public get connectedClients(): Client[] {
        return [...this.clients.values()];
    }
    public get players(): Client[] {
        return this.connectedClients.filter(c => this.checkClientIsPlayer(c));
    }
    public get gameOptions(): GameOption {
        return { ...this._gameOptions };
    }
    public get status(): RoomState {
        return this._status;
    }
    public get hasBot(): boolean {
        return Boolean(this.bot);
    }
    public get turn(): number {
        return this._turn;
    }
    public get closed(): boolean {
        return this._closed;
    }
    public get joinGameBuffer(): Buffer {
        return this.recorder.joinGameDataBuffer;
    }
    public get tpClient(): Client {
        return this._tpClient;
    }
    public get type(): RoomType {
        return this._type;
    }
    public get maximumPlayerCount(): number {
        return this._gameOptions.type === GameType.Tag ? 4 : 2;
    }

    private createYGOProServerInstance = () => {
        if (process.env.SERV_PORT) {
            this.portNumber = parseInt(process.env.SERV_PORT, 10);
            return Promise.resolve();
        }

        return new Promise<void>((resolve, reject) => {
            const param = [
                "0",
                this._gameOptions.lfList.toString(), // 0
                this._gameOptions.rule.toString(), // 0
                this._gameOptions.type, // 1
                this._gameOptions.duelRule.toString(), // F
                this._gameOptions.noDeckCheck ? "T" : "F", // F
                this._gameOptions.noDeckShuffle ? "T" : "F", // F
                this._gameOptions.lifePoints.toString(), // 8000
                this._gameOptions.startHandCount.toString(), // 5
                this._gameOptions.drawCount.toString(), // 1
                this._gameOptions.timeLimit.toString(), // 180
                "0",
            ];

            this.serverProcess = ChildProcess.spawn("./ygopro", param, {
                cwd: YGOPRO_PATH,
            });

            this.serverProcess.on("error", (e: Error) => {
                this.release();
                reject(e);
            });
            this.serverProcess.on("exit", () => {
                this.release();
            });

            this.serverProcessId = this.serverProcess.pid;
            this.serverProcess.stdout.setEncoding("utf8");
            this.serverProcess.stdout.on("data", (data: string) => {
                if (!this.portNumber) {
                    this.portNumber = parseInt(data, 10);
                    logger.info(`YGOPro server opened with port ${this.portNumber} with room id: ${this.id}`);

                    if (this.bot) {
                        setTimeout(() => {
                            this.inviteBot();
                        }, 200);
                    }

                    this.recorder.connectTo(this);
                    resolve();
                }
            });
        });
    };
    public start = async () => {
        if (this.isStarted) {
            throw new Error("Cannot start an instance of server twice!");
        }

        await this.createYGOProServerInstance();
    };

    private inviteBot = () => {
        if (!this.bot) {
            return;
        }

        const botData = {
            name: this.bot.name,
            deck: encodeURIComponent(this.bot.deck),
            dialog: encodeURIComponent(this.bot.dialog),
            host: "127.0.0.1",
            version: 0x1351,
            password: this.id,
            port: 3000,
        };

        request(
            {
                url: `http://localhost:3001/?${QueryString.stringify(botData)}`,
            },
            error => {
                if (error) {
                    logger.warn(`windbot add error from ${this.id}: ${error}`);
                    this.broadcastChat(
                        "[Server]: 봇을 불러오는데 실패 하였습니다. 관리자에게 문의하세요.",
                        ChatType.RED,
                    );
                }
            },
        );
    };
    private startHostTimeout = (time: number = 15) => {
        if (this.status !== RoomState.Ready || this.readiedPlayerCountWithoutHost < this.maximumPlayerCount - 1) {
            return;
        }

        if (time) {
            if (!(time % 5)) {
                this.broadcastChat(
                    `모든 플레이어가 준비 하였습니다. 방장은 ${time}초 내로 시작하지 않을시 강제 퇴장 됩니다.`,
                    ChatType.LIGHTBLUE,
                );
            }

            setTimeout(() => {
                this.startHostTimeout(time - 1);
            }, 1000);
        } else {
            this.players.forEach(player => {
                if (player.isHost) {
                    player.close(CloseReason.TimeoutKicked, true);
                }
            });
        }
    };

    public broadcastChat = (message: string, color: ChatType, record: boolean = false) => {
        if (record) {
            const packet = PacketBuilder.build(PacketType.ServerToClient)
                .type(ServerToClientMessageType.CHAT)
                .add("unsigned short", color)
                .add("string", message, 255)
                .build();

            this.recorder.record(packet);
        }

        this.clients.forEach(client => {
            client.sendChat(message, color);
        });
    };
    public selectTP = (client: Client) => {
        if (!this.checkClientIsPlayer(client)) {
            return;
        }

        this._status = RoomState.FirstGo;
        this._tpClient = client;
    };
    public changeSide = (from: Client) => {
        if (from.position === 0) {
            this._status = RoomState.Siding;
        }
    };
    public replacePlayer = (client: Client, target: Client) => {
        this.clients.delete(client);
        this.clients.add(target);
    };

    public checkClientIsPostSpectator(client: Client) {
        return this.recorder.isClientPiped(client);
    }
    public checkClientIsPlayer(client: Client) {
        return this.clients.has(client) && client.position >= 0 && client.position < this.maximumPlayerCount;
    }
    public checkClientIsSpectator(client: Client) {
        return client.position >= 7 || this.checkClientIsPostSpectator(client);
    }

    public connect = (client: Client) => {
        const isPostSpectator = this.status !== RoomState.Ready;
        this.clients.add(client);
        this.clients.forEach(c => {
            let message;
            if (client === c) {
                if (isPostSpectator) {
                    message = "[Server]: 관전자로서 입장하셨습니다.";
                } else {
                    message = "[Server]: 게임에 입장하셨습니다.";
                }
            } else if (isPostSpectator) {
                message = `[Server]: ${client.name}님이 관전자로 입장하셨습니다.`;
            } else {
                message = `[Server]: ${client.name}님이 입장하셨습니다.`;
            }

            c.sendChat(message, ChatType.GREEN);
        });

        if (isPostSpectator) {
            this.recorder.pipe(client);
        }
    };
    public disconnect = (client: Client, reason: CloseReason) => {
        if (!this.clients.has(client)) {
            return;
        }

        const isPostSpectator = this.recorder.isClientPiped(client);
        this.recorder.unpipe(client);

        this.clients.delete(client);
        if (this.clients.size && !(this.hasBot && client.isHost)) {
            let name = `${client.name}`;
            if (isPostSpectator) {
                name = `관전자 ${name}`;
            }

            this.clients.forEach(c => {
                if (client !== c) {
                    switch (reason) {
                        case CloseReason.Heartbeat:
                        case CloseReason.Timeout:
                        case CloseReason.Errored:
                            c.sendChat(`[Server]: ${name}님의 연결이 끊겼습니다.`, ChatType.RED);
                            break;

                        case CloseReason.Kicked:
                            c.sendChat(`[Server]: ${name}님이 강제 퇴장 당하셨습니다.`, ChatType.YELLOW);
                            break;

                        case CloseReason.TimeoutKicked:
                            c.sendChat(
                                `[Server]: ${name}님이 게임을 시작하지 않아 강제 퇴장 당하셨습니다.`,
                                ChatType.RED,
                            );
                            break;

                        case CloseReason.ReconnectTimeout:
                            c.sendChat(
                                `[Server]: ${name}님이 제한 시간동안 재접속 하지 않았으므로 패배 처리됩니다.`,
                                ChatType.RED,
                            );
                            break;

                        default:
                            c.sendChat(`[Server]: ${name}님이 퇴장하셨습니다.`, ChatType.GREEN);
                            break;
                    }
                }
            });
        } else if (this.serverProcess) {
            this.serverProcess.kill();
            this.release();
        }

        const reconnectPlugin = Reconnect.get(client);
        if (reconnectPlugin) {
            reconnectPlugin.unregister(false, true);
        }
    };

    private onRecorderReceive = (packet: Packet<PacketType.ServerToClient>) => {
        const packetData = packet.process();
        switch (packetData.type) {
            case ServerToClientMessageType.SELECT_HAND:
                this.onSelectHand();
                break;

            case ServerToClientMessageType.DUEL_START:
                this.onDuelStart();
                break;

            case ServerToClientMessageType.HS_PLAYER_CHANGE:
                this.onPlayerChange(packetData);
                break;

            default:
                break;
        }
    };
    private onRecorderEnd = () => {
        Uploader.upload(this, this.players, this.recorder);
    };
    private onSelectHand = () => {
        this._status = RoomState.ChoosingFirst;
    };
    private onDuelStart = () => {
        if (this.status === RoomState.Ready) {
            this._status = RoomState.ChoosingFirst;
            this._turn = 0;
        }
    };
    private onPlayerChange = ({ status: data }: PlayerChangeMessageData) => {
        const position = data >> 4;
        const isReady = (data & 0xf) === 9;
        if (position < this.maximumPlayerCount) {
            this.readiedPlayerCountWithoutHost = 0;
            this.players.forEach(client => {
                if (!client.isHost && client.position === position && isReady) {
                    this.readiedPlayerCountWithoutHost++;
                }
            });

            if (this.readiedPlayerCountWithoutHost >= this.maximumPlayerCount - 1) {
                this.startHostTimeout();
            }
        }
    };

    public processGameMessage = (type: GameMessageType, data: Buffer, from: Client) => {
        switch (type) {
            case GameMessageType.START:
                if (from.position === 0) {
                    this._turn = 0;
                }

                this._status = RoomState.Dueling;
                break;

            case GameMessageType.WIN:
                this._status = RoomState.Ended;
                this._turn = 0;
                break;

            case GameMessageType.NEW_TURN:
                if (from.position === 0) {
                    this._turn++;
                }
                break;

            default:
                break;
        }
    };

    public destroy = () => {
        this.connectedClients.forEach(client => {
            client.sendChat("[Server]: 서버에 의해 방이 닫혔습니다.", ChatType.RED);
            client.close(CloseReason.Destroyed, true);
        });
    };
    public release = () => {
        this.clients.forEach(client => {
            if (client.isAI) {
                client.close(CloseReason.Unknown);
            }
        });

        this.recorder.release();
        this.clients.clear();
        this._closed = true;

        ConnectionManager.clearConnectionsByRoom(this);

        clearInterval(this.heartbeatInterval);
        delete Room.openedRooms[this.id];
    };

    public toData() {
        const object = new RoomObject();
        object.id = this.id;
        object.type = this.type;
        object.status = this.status;
        object.connectedClients = [...this.clients.values()].map(c => c.toData());

        return object;
    }
}
