import net from "net";
import { EventEmitter } from "tsee";

import Room from "@game/Room";

import Client from "@network/Client";
import PacketBuilder from "@network/PacketBuilder";
import { PacketType } from "@network/message";
import { ClientToServerMessageType } from "@network/message/ctos";
import Packet from "@network/Packet";
import { ServerToClientMessageType } from "@network/message/stoc";
import moment from "moment";

type RecorderEvents = {
    receive: (packet: Packet<PacketType.ServerToClient>) => void;
    end: () => void;
};

export default class Recorder extends EventEmitter<RecorderEvents> {
    private recorderSocket: net.Socket = new net.Socket();
    private spectatorSocket: net.Socket = new net.Socket();
    private recordedData: Buffer[] = [];
    private spectatorData: Buffer[] = [];
    private pipeClients: Set<Client> = new Set<Client>();
    private _joinGameDataBuffer: Buffer;
    private _startTime: moment.Moment;

    public get joinGameDataBuffer(): Buffer {
        return this._joinGameDataBuffer.slice(0);
    }
    public get recordedBuffer(): Buffer {
        return Buffer.concat(this.recordedData);
    }
    public get startTime(): moment.Moment {
        return this._startTime;
    }

    public connectTo = (room: Room) => {
        this.spectatorSocket.connect(room.port, this.onSpectatorConnect);
        this.spectatorSocket.on("data", this.onSpectatorData);
        this.spectatorSocket.on("error", this.onError);

        this.recorderSocket.connect(room.port, this.onConnect);
        this.recorderSocket.on("data", this.onData);
        this.recorderSocket.on("error", this.onError);
    };

    public isClientPiped = (client: Client) => {
        return this.pipeClients.has(client);
    };
    public pipe = (client: Client) => {
        this.spectatorData.forEach(data => {
            client.send(data);
        });

        this.pipeClients.add(client);
    };
    public unpipe = (client: Client) => {
        this.pipeClients.delete(client);
    };

    private sendMockedPackets = (socket: net.Socket, name: string) => {
        const data = [
            PacketBuilder.build(PacketType.ClientToServer)
                .type(ClientToServerMessageType.PLAYER_INFO)
                .add("string", name)
                .build(),

            PacketBuilder.build(PacketType.ClientToServer)
                .type(ClientToServerMessageType.JOIN_GAME)
                .add("unsigned short", 4945)
                .add("unsigned short", 0)
                .add("unsigned int", 0)
                .add("string", name)
                .build(),

            PacketBuilder.build(PacketType.ClientToServer)
                .type(ClientToServerMessageType.HS_TOOBSERVER)
                .build(),
        ];

        data.forEach(packet => socket.write(packet.buffer));
    };

    private onConnect = () => {
        this.sendMockedPackets(this.recorderSocket, "Marshtomp");
    };
    private onData = (buffer: Buffer) => {
        const packets = Packet.parse(buffer, PacketType.ServerToClient);
        if (!this._joinGameDataBuffer) {
            packets.some(packet => {
                if (packet.type === ServerToClientMessageType.JOIN_GAME) {
                    this._joinGameDataBuffer = packet.dataBuffer;
                    return true;
                }

                return false;
            });
        }

        packets.forEach(packet => {
            this.emit("receive", packet);
        });

        this.recordedData.push(buffer);

        if (packets.some(packet => packet.type === 22)) {
            this.emit("end");
        } else if (packets.some(packet => packet.type === ServerToClientMessageType.DUEL_START)) {
            this._startTime = moment();
        }
    };
    private onError = () => {};

    private onSpectatorConnect = () => {
        this.sendMockedPackets(this.spectatorSocket, "the Big Brother");
    };
    private onSpectatorData = (buffer: Buffer) => {
        this.pipeClients.forEach(client => {
            client.send(buffer);
        });

        this.spectatorData.push(buffer);
    };

    public record = (packet: Packet<PacketType.ServerToClient>) => {
        this.recordedData.push(packet.buffer);
        this.spectatorData.push(packet.buffer);
    };

    public release() {
        this.recorderSocket.destroy();
        this.spectatorSocket.destroy();

        this.recordedData = [];
        this.spectatorData = [];

        this.pipeClients.clear();
    }
}
