import * as _ from "lodash";

import Packet from "@network/Packet";
import PacketBuilder from "@network/PacketBuilder";
import { PacketType } from "@network/message";
import { ServerToClientMessageType } from "@network/message/stoc";
import { ClientToServerMessageType } from "@network/message/ctos";

import Client, { CloseReason } from "@network/Client";
import Bridge from "@network/Bridge";

import ConnectionManager, { Connection } from "@plugins/reconnect/ConnectionManager";

import { ChatType, RoomState } from "@root/constants";
import { GameType, PacketData } from "@root/types";
import { RoomType } from "@game/Room";

interface RoomReconnectionData {
    newPosition: number;
}

export default class Reconnect {
    private static readonly instances: Map<Client, Reconnect> = new Map<Client, Reconnect>();

    public static install(client: Client, server: Bridge) {
        let reconnect = Reconnect.instances.get(client);
        if (!reconnect) {
            reconnect = new Reconnect(client, server);
            Reconnect.instances.set(client, reconnect);
        }

        return reconnect;
    }
    public static get(client: Client) {
        return Reconnect.instances.get(client);
    }

    private readonly client: Client;
    private readonly server: Bridge;
    private _hadNewConnection: boolean;
    private timeConfirmRequired: boolean;
    private beforeReconnecting: boolean;
    private reconnecting: boolean;

    private constructor(client: Client, server: Bridge) {
        this.client = client;
        this.server = server;
        this.reconnecting = false;
        this.beforeReconnecting = false;
        this._hadNewConnection = false;
        this.timeConfirmRequired = false;

        this.client.on("send", this.onSend);
        this.client.on("receive", this.onReceive);
    }

    public get hadNewConnection() {
        return this._hadNewConnection;
    }
    public get isReconnectedToRoom() {
        return this.beforeReconnecting;
    }

    private onSend = (__: Packet<PacketType.ClientToServer>, packetData: PacketData<PacketType.ClientToServer>) => {
        switch (packetData.type) {
            case ClientToServerMessageType.TIME_CONFIRM:
                this.onTimeConfirm();
                break;

            default:
                break;
        }
    };
    private onReceive = (__: Packet<PacketType.ServerToClient>, packetData: PacketData<PacketType.ServerToClient>) => {
        switch (packetData.type) {
            case ServerToClientMessageType.TIME_LIMIT:
                this.onTimeLimit();
                break;

            default:
                break;
        }
    };

    private onTimeLimit = () => {
        if (this.client.closed) {
            this.sendTimeConfirm();
        } else {
            this.timeConfirmRequired = true;
        }
    };
    private onTimeConfirm = () => {
        this.timeConfirmRequired = false;
    };

    private sendTimeConfirm = () => {
        this.server.send(
            PacketBuilder.build(PacketType.ClientToServer)
                .type(ClientToServerMessageType.TIME_CONFIRM)
                .build(),
        );
    };

    public register = () => {
        const room = this.client.currentRoom;
        if (this._hadNewConnection) {
            return false;
        }

        if (
            !room ||
            // client.system_kicked ||
            // client.flee_free ||
            ConnectionManager.get(this.client) ||
            room.checkClientIsPostSpectator(this.client) ||
            !room.checkClientIsPlayer(this.client) ||
            room.status === RoomState.Ready ||
            room.hasBot ||
            room.type === RoomType.Random
            // (settings.modules.reconnect.auto_surrender_after_disconnect && room.hostinfo.mode !== 1) ||
        ) {
            return false;
        }

        const connection: Connection = {
            room,
            client: this.client,
            server: this.server,
            deck: this.client.deckBuffer,
        };

        connection.timeout = setTimeout(() => {
            room.disconnect(this.client, CloseReason.ReconnectTimeout);
            connection.server.close(true);
        }, 15000);

        ConnectionManager.set(this.client, connection);

        room.broadcastChat(
            `[Server]: ${this.client.name}님의 연결이 끊겼습니다. 15초 내에 재접속 하지 않을시 패배 처리 됩니다.`,
            ChatType.YELLOW,
        );

        if (this.timeConfirmRequired) {
            this.timeConfirmRequired = false;
            this.sendTimeConfirm();
        }

        /*
            if (
                settings.modules.reconnect.auto_surrender_after_disconnect &&
                room.duel_stage === ygopro.constants.DUEL_STAGE.DUELING
            ) {
                ygopro.ctos_send(client.server, "SURRENDER");
            }
        */

        return true;
    };
    public unregister = (exact?: boolean, reconnected?: boolean) => {
        const connection = ConnectionManager.get(this.client);
        if (connection) {
            if (exact && connection.client !== this.client) {
                return false;
            }

            ConnectionManager.release(this.client, reconnected);
            ConnectionManager.remove(this.client);
            return true;
        }

        return false;
    };
    public checkReconnectable = (deck?: Buffer) => {
        // if (client.system_kicked) {
        //     return false;
        // }

        const connection = ConnectionManager.get(this.client);
        if (!connection) {
            return false;
        }

        if (connection.room.closed) {
            this.unregister();
            return false;
        }

        return !(deck && !_.isEqual(deck, connection.deck));
    };

    private sendDuelStartMessage = () => {
        this.client.send(
            PacketBuilder.build(PacketType.ServerToClient)
                .type(ServerToClientMessageType.DUEL_START)
                .build(),
        );
    };

    private sendRoomReconnectingData = () => {
        const connection = ConnectionManager.get(this.client);
        if (!connection) {
            return false;
        }

        this.client.sendChat(
            "[Server]: 이전 게임에 다시 입장하셨습니다. 이전 진행 상황부터 다시시작하시려면 같은 덱을 선택해주세요.",
            ChatType.GREEN,
        );
        this.client.send(
            PacketBuilder.build(PacketType.ServerToClient)
                .type(ServerToClientMessageType.JOIN_GAME)
                .buffer(connection.room.joinGameBuffer)
                .build(),
        );

        let newPosition = connection.client.position;
        if (connection.client.isHost) {
            newPosition += 0x10;
        }

        this.client.send(
            PacketBuilder.build(PacketType.ServerToClient)
                .type(ServerToClientMessageType.TYPE_CHANGE)
                .add("unsigned char", newPosition)
                .build(),
        );

        connection.room.connectedClients.forEach(client => {
            this.client.send(
                PacketBuilder.build(PacketType.ServerToClient)
                    .type(ServerToClientMessageType.HS_PLAYER_ENTER)
                    .add("string", client.name, 20)
                    .add("unsigned char", client.position)
                    .build(),
            );
        });

        return true;
    };
    private sendGameReconnectingData = () => {
        const connection = ConnectionManager.get(this.client);
        if (!connection) {
            return;
        }

        this.reconnecting = true;
        this.client.sendChat("[System]: 이전 게임에 재접속 하였습니다.", ChatType.GREEN);

        switch (connection.room.status) {
            case RoomState.ChoosingFirst:
                this.sendDuelStartMessage();
                if (
                    connection.room.gameOptions.type !== GameType.Tag ||
                    this.client.position === 0 ||
                    this.client.position === 2
                ) {
                    this.client.send(
                        PacketBuilder.build(PacketType.ServerToClient)
                            .type(ServerToClientMessageType.SELECT_HAND)
                            .build(),
                    );
                }
                this.reconnecting = false;
                break;

            case RoomState.FirstGo:
                this.sendDuelStartMessage();
                if (this.client === connection.room.tpClient) {
                    this.client.send(
                        PacketBuilder.build(PacketType.ServerToClient)
                            .type(ServerToClientMessageType.SELECT_TP)
                            .build(),
                    );
                }
                this.reconnecting = false;
                break;

            case RoomState.Siding:
                this.sendDuelStartMessage();
                if (!this.client.selectedPreduel) {
                    this.client.send(
                        PacketBuilder.build(PacketType.ServerToClient)
                            .type(ServerToClientMessageType.CHANGE_SIDE)
                            .build(),
                    );
                }
                this.reconnecting = false;
                break;

            default:
                connection.server.send(
                    PacketBuilder.build(PacketType.ClientToServer)
                        .type(ClientToServerMessageType.REQUEST_FIELD)
                        .build(),
                );
                break;
        }
    };

    public reconnectToRoom = (): RoomReconnectionData | null => {
        if (this.checkReconnectable()) {
            const result: RoomReconnectionData = {} as any;
            const connection = ConnectionManager.get(this.client);
            if (!connection) {
                return null;
            }

            this.beforeReconnecting = true;
            result.newPosition = connection.client.position;

            if (this.sendRoomReconnectingData()) {
                return null;
            }
        }

        return null;
    };
    public reconnectToGame = () => {
        if (!this.checkReconnectable()) {
            return false;
        }

        this.beforeReconnecting = false;

        const connection = ConnectionManager.get(this.client);
        if (!connection) {
            return false;
        }

        const oldReconnect = Reconnect.get(connection.client);
        if (!oldReconnect) {
            return false;
        }

        const oldServer = this.client.replaceServer(connection.server);
        oldServer.close(true, true);

        this.client.replaceClient(connection.client);
        oldReconnect._hadNewConnection = true;

        this.sendGameReconnectingData();

        connection.room.broadcastChat(`[Server]: ${this.client.name}님이 게임에 재접속 하였습니다.`, ChatType.GREEN);

        this.unregister(false, true);
        return true;
    };
}
