import { BaseMessageData, PacketType } from "@network/message";
import { ClientToServerMessages, ClientToServerMessageType } from "@network/message/ctos";
import { ServerToClientMessages, ServerToClientMessageType } from "@network/message/stoc";

import { UnionToIntersection } from "@root/types";

export type PacketDataType = "unsigned int" | "unsigned short" | "unsigned char" | "string";

export interface MessageStructItem<Names> {
    name: Names;
    type: PacketDataType;
    length?: number;
    offset?: number;
}

export type MessageStructureContainer = {
    [PacketType.ClientToServer]: {
        [Key in ClientToServerMessages["type"]]: MessageStructItem<
            Exclude<keyof UnionToIntersection<ClientToServerMessages>, keyof BaseMessageData<any>>
        >[];
    };

    [PacketType.ServerToClient]: {
        [Key in ServerToClientMessages["type"]]: MessageStructItem<
            Exclude<keyof UnionToIntersection<ServerToClientMessages>, keyof BaseMessageData<any>>
        >[];
    };
};

const MESSAGE_STRUCTURE: MessageStructureContainer = {
    [PacketType.ClientToServer]: {
        [ClientToServerMessageType.CHAT]: [{ name: "message", type: "string", length: 255 }],
        [ClientToServerMessageType.UPDATE_DECK]: [],
        [ClientToServerMessageType.HAND_RESULT]: [],
        [ClientToServerMessageType.TP_RESULT]: [],
        [ClientToServerMessageType.TIME_CONFIRM]: [],
        [ClientToServerMessageType.HS_KICK]: [{ name: "position", type: "unsigned char" }],
        [ClientToServerMessageType.PLAYER_INFO]: [{ name: "name", type: "string", length: 20 }],
        [ClientToServerMessageType.JOIN_GAME]: [
            { name: "version", type: "unsigned short" },
            { name: "align", type: "unsigned short" },
            { name: "gameid", type: "unsigned int" },
            { name: "password", type: "string", length: 20 },
        ],
    },
    [PacketType.ServerToClient]: {
        [ServerToClientMessageType.HS_PLAYER_CHANGE]: [{ name: "status", type: "unsigned char" }],
        [ServerToClientMessageType.SELECT_TP]: [],
        [ServerToClientMessageType.CHANGE_SIDE]: [],
        [ServerToClientMessageType.SELECT_HAND]: [],
        [ServerToClientMessageType.DUEL_START]: [],
        [ServerToClientMessageType.JOIN_GAME]: [],
        [ServerToClientMessageType.TIME_LIMIT]: [
            { name: "player", type: "unsigned char" },
            { name: "timeLeft", type: "unsigned short" },
        ],
        [ServerToClientMessageType.GAME_MSG]: [],
        [ServerToClientMessageType.TYPE_CHANGE]: [{ name: "value", type: "unsigned char" }],
    },
};

export default function getMessageStructure<
    PT extends keyof MessageStructureContainer,
    MT extends keyof MessageStructureContainer[PT]
>(packetType: PT, messageType: MT): MessageStructItem<any>[] {
    return (MESSAGE_STRUCTURE as any)[packetType][messageType];
}
